<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\frontend\CourseController;
use App\Http\Controllers\ApiController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\Api\DealsMessagesController;
use App\Http\Controllers\Admin\AdminOrderController;
use App\Http\Controllers\Admin\DisputeController;
use App\Http\Controllers\Api\JobController;
use App\Http\Controllers\Api\BanController;
use App\Http\Controllers\Api\CertificateController;
use App\Http\Controllers\ReportController;


/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware("auth:sanctum")->get("/user", function (Request $request) {
    return $request->user();
});

Route::middleware('auth:sanctum')->get('/certificate-status', [CertificateController::class, 'getCertificateStatus']);
Route::get('/certificate/{identifier}', [CertificateController::class, 'show'])->name('certificate');
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/certificates', [CertificateController::class, 'getUserCertificates']);
    Route::post('/certificates', [CertificateController::class, 'createCertificate']);
});
Route::middleware('auth:sanctum')->get(
    '/certificates/{identifier}/download',
    [CertificateController::class, 'downloadCertificate']
);
Route::middleware('auth:sanctum')->post(
    '/generate-certificate',
    [CertificateController::class, 'generateCertificate']
);

Route::post("/login", [ApiController::class, "login"]);
Route::post("/signup", [ApiController::class, "signup"]);

Route::group(["middleware", ["auth:sanctum"]], function () {
    Route::get("/top_courses", [ApiController::class, "top_courses"]);
    Route::get("/all_categories", [ApiController::class, "all_categories"]);
    Route::get("/categories", [ApiController::class, "categories"]);
    Route::get("/category_details", [ApiController::class, "category_details"]);
    Route::get("/sub_categories/{id}", [ApiController::class, "sub_categories"]);
    Route::get("/category_wise_course", [ApiController::class, "category_wise_course"]);
    Route::get("/category_subcategory_wise_course", [ApiController::class, "category_subcategory_wise_course"]);
    Route::get("/filter_course", [ApiController::class, "filter_course"]);
    Route::get("/my_wishlist", [ApiController::class, "my_wishlist"]);
    Route::get("/toggle_wishlist_items", [ApiController::class, "toggle_wishlist_items"]);
    Route::get("/languages", [ApiController::class, "languages"]);
    Route::get("/courses_by_search_string", [ApiController::class, "courses_by_search_string"]);
    Route::get("/my_courses", [ApiController::class, "my_courses"]);
    Route::get("/sections", [ApiController::class, "sections"]);
    Route::get("/course_details_by_id", [ApiController::class, "course_details_by_id"]);
    Route::post("/update_password", [ApiController::class, "update_password"]);
    Route::post("/update_userdata", [ApiController::class, "update_userdata"]);
    Route::post("/account_disable", [ApiController::class, "account_disable"]);
    Route::get("/cart_list", [ApiController::class, "cart_list"]);
    Route::get("/toggle_cart_items", [ApiController::class, "toggle_cart_items"]);
    Route::get("/save_course_progress", [ApiController::class, "save_course_progress"]);
    Route::post("/logout", [ApiController::class, "logout"]);

    Route::middleware("auth:sanctum")->group(function () {
        Route::get("/orders", [OrderController::class, "index"]);
        Route::post("/orders", [OrderController::class, "store"]);
    });

    Route::prefix("admin")->group(function () {
        Route::get("/orders", [AdminOrderController::class, "index"])->name("admin.orders.index");
        Route::put("/orders/{id}/status", [AdminOrderController::class, "updateStatus"])->name("admin.orders.updateStatus");
        Route::delete("/orders/{id}", [AdminOrderController::class, "destroy"])->name("admin.orders.destroy");
    });

    // مسارات الرسائل القديمة (تم تصحيحها لتتوافق مع Controller)
    Route::middleware("auth:sanctum")->group(function () {
        Route::get("/get_user_threads", [DealsMessagesController::class, "getUserThreads"]);
        Route::post("/get_thread_id/{receiver_id}", [DealsMessagesController::class, "openapp"]);
        // **تم تصحيح هذا المسار: يجب أن يستقبل thread_id كمعامل في المسار**
        Route::get("/deals_messages/{thread_id}", [DealsMessagesController::class, "getMessages"]); // جلب الرسائل
        Route::post("/deals_messages/send", [DealsMessagesController::class, "sendMessage"]); // إرسال رسالة
        Route::post("/deals_messages/mark-as-read", [DealsMessagesController::class, "markAsRead"]); // تعيين كمقروءة
    });

    Route::middleware(["auth:api"])->group(function () {
        Route::prefix("wallet")->group(function () {
            // الحصول على معلومات المحفظة
            Route::get("/", function (Request $request) {
                $user = $request->user();

                $transactions = \App\Models\WalletTransaction::where("sender_id", $user->id)
                    ->orWhere("receiver_id", $user->id)
                    ->with(["sender", "receiver"])
                    ->orderBy("created_at", "desc")
                    ->take(10)
                    ->get();

                return response()->json([
                    "wallet_id" => $user->wallet_id,
                    "available_balance" => $user->wallet_balance,
                    "frozen_balance" => $user->frozen_balance,
                    "transactions" => $transactions,
                ]);
            });

            // إرسال الأموال
            Route::post("/send", function (Request $request) {
                $request->validate([
                    "receiver_id" => "required|exists:users,wallet_id",
                    "amount" => "required|numeric|min:0.01",
                ]);

                $sender = $request->user();
                $receiver = \App\Models\User::where("wallet_id", $request->receiver_id)->first();

                if ($sender->id === $receiver->id) {
                    return response()->json(["message" => "لا يمكن إرسال الأموال لنفسك"], 400);
                }

                if ($sender->wallet_balance < $request->amount) {
                    return response()->json(["message" => "رصيدك غير كافي لإتمام هذه العملية"], 400);
                }

                \DB::transaction(function () use ($sender, $receiver, $request) {
                    $sender->wallet_balance -= $request->amount;
                    $sender->save();

                    $receiver->frozen_balance += $request->amount;
                    $receiver->save();

                    \App\Models\WalletTransaction::create([
                        "transaction_id" => "TXN" . \Illuminate\Support\Str::upper(\Illuminate\Support\Str::random(10)),
                        "sender_id" => $sender->id,
                        "receiver_id" => $receiver->id,
                        "amount" => $request->amount,
                        "status" => "pending",
                        "notes" => "انتظار تأكيد من المرسل",
                    ]);
                });

                return response()->json(["message" => "تم إرسال الأموال بنجاح"]);
            });

            // تأكيد المعاملة
            Route::post("/confirm/{transactionId}", function (Request $request, $transactionId) {
                $transaction = \App\Models\WalletTransaction::where("transaction_id", $transactionId)
                    ->where("sender_id", $request->user()->id)
                    ->where("status", "pending")
                    ->firstOrFail();

                \DB::transaction(function () use ($transaction) {
                    $receiver = \App\Models\User::find($transaction->receiver_id);

                    $receiver->frozen_balance -= $transaction->amount;
                    $receiver->wallet_balance += $transaction->amount;
                    $receiver->save();

                    $transaction->status = "completed";
                    $transaction->notes = "تم التأكيد من قبل المرسل";
                    $transaction->save();
                });

                return response()->json(["message" => "تم تأكيد الإرسال بنجاح"]);
            });

            // إلغاء المعاملة
            Route::post("/cancel/{transactionId}", function (Request $request, $transactionId) {
                $transaction = \App\Models\WalletTransaction::where("transaction_id", $transactionId)
                    ->where("sender_id", $request->user()->id)
                    ->where("status", "pending")
                    ->firstOrFail();

                \DB::transaction(function () use ($transaction, $request) {
                    $sender = $request->user();
                    $receiver = \App\Models\User::find($transaction->receiver_id);

                    $sender->wallet_balance += $transaction->amount;
                    $sender->save();

                    $receiver->frozen_balance -= $transaction->amount;
                    $receiver->save();

                    $transaction->status = "cancelled";
                    $transaction->notes = "تم الإلغاء من قبل المرسل";
                    $transaction->save();
                });

                return response()->json(["message" => "تم إلغاء العملية بنجاح"]);
            });

            // عرض معاملات السحب الخاصة بالمستخدم
            Route::get("/withdrawals", function (Request $request) {
                $user = $request->user(); // الحصول على المستخدم المسجل الدخول

                // جلب جميع عمليات السحب الخاصة بالمستخدم الحالي
                $withdrawals = \App\Models\WithdrawalRequest::where("user_id", $user->id)
                    ->orderBy("created_at", "desc") // ترتيب من الأحدث إلى الأقدم
                    ->get();

                return response()->json($withdrawals);
            });
            // طلب السحب
            Route::post("/withdraw", function (Request $request) {
                $request->validate([
                    "amount" => "required|numeric|min:0.01",
                    "method" => "required|string",
                    "details" => "required|string",
                ]);

                $user = $request->user();

                if ($user->wallet_balance < $request->amount) {
                    return response()->json(["message" => "رصيدك غير كافي لتنفيذ عملية السحب"], 400);
                }

                \DB::transaction(function () use ($user, $request) {
                    $user->wallet_balance -= $request->amount;
                    $user->save();

                    \App\Models\WithdrawalRequest::create([
                        "request_id" => "WDR" . \Illuminate\Support\Str::upper(\Illuminate\Support\Str::random(10)),
                        "user_id" => $user->id,
                        "amount" => $request->amount,
                        "withdrawal_method" => $request->method,
                        "withdrawal_details" => ["details" => $request->details],
                        "status" => "pending",
                    ]);
                });

                return response()->json(["message" => "تم تقديم طلب السحب بنجاح"]);
            });

            // إضافة أموال
            Route::post("/add", function (Request $request) {
                $request->validate([
                    "wallet_id" => "required|exists:users,wallet_id",
                    "amount" => "required|numeric|min:0.01",
                ]);

                $sender = $request->user();
                $receiver = \App\Models\User::where("wallet_id", $request->wallet_id)->first();

                \DB::transaction(function () use ($sender, $receiver, $request) {
                    $receiver->wallet_balance += $request->amount;
                    $receiver->save();

                    \App\Models\WalletTransaction::create([
                        "transaction_id" => "ADD" . \Illuminate\Support\Str::upper(\Illuminate\Support\Str::random(10)),
                        "sender_id" => $sender->id,
                        "receiver_id" => $receiver->id,
                        "amount" => $request->amount,
                        "status" => "completed",
                        "notes" => "إضافة رصيد من قبل المستخدم",
                    ]);
                });

                return response()->json(["message" => "تم إضافة الأموال بنجاح"]);
            });
        });
    });
Route::middleware('auth:api')->group(function () {
    
    // إنشاء إبلاغ جديد
    Route::post('/reports', [ReportController::class, 'store']);
    
    // التحقق من وجود إبلاغ نشط للمحادثة
    Route::get('/reports/check-active/{conversationId}', [ReportController::class, 'checkActiveReport']);
    
    // التحقق من إمكانية الإبلاغ عن المحادثة
    Route::get('/reports/can-report/{conversationId}', [ReportController::class, 'canReport']);
    
    // التحقق من حالة المحادثة (مغلقة أم لا)
    Route::get('/conversations/{conversationId}/status', [ReportController::class, 'checkConversationStatus']);
    
    // الحصول على إبلاغات المستخدم الحالي
    Route::get('/my-reports', [ReportController::class, 'getUserReports']);
    
    // الحصول على إحصائيات الإبلاغات (للإدارة فقط)
    Route::get('/reports/statistics', [ReportController::class, 'getStatistics'])
        ->middleware('can:view-admin-panel');
});

// Routes إضافية للتوافق مع الكود الموجود
Route::middleware('auth:api')->group(function () {
    
    // التحقق من وجود نزاع نشط (للتوافق مع الكود الموجود)
    Route::get('/{conversationId}/has-dispute', function($conversationId) {
        try {
            $hasDispute = \App\Models\Report::hasActiveReport($conversationId);
            return response()->json(['has_dispute' => $hasDispute]);
        } catch (\Exception $e) {
            return response()->json(['has_dispute' => false], 500);
        }
    });
    
    // إنشاء نزاع (للتوافق مع الكود الموجود)
    Route::post('/disputes', [ReportController::class, 'store']);
    
});



    // مسارات الرسائل الجديدة (مطابقة لـ web.php)
    Route::middleware(["auth:sanctum"])->prefix("instructor")->name("api.instructor.")->group(function () {
        Route::prefix("conversations")->name("conversations.")->group(function () {
            Route::get("/", [DealsMessagesController::class, "index"])->name("index");
            Route::get("/{thread_id}", [DealsMessagesController::class, "show"])->name("show");
            Route::post("/start", [DealsMessagesController::class, "startConversation"])->name("start");
            Route::post("/send-message", [DealsMessagesController::class, "sendMessage"])->name("send-message");
            Route::get("/open/{receiver_id}", [DealsMessagesController::class, "openOrCreateConversation"])->name("open");
        });
    });

    Route::middleware(["auth:sanctum"])->prefix("api/conversations")->name("api.conversations.")->group(function () {
        Route::get("/{thread_id}/messages", [DealsMessagesController::class, "getMessages"])->name("messages");
        Route::post("/{thread_id}/send", [DealsMessagesController::class, "sendMessage"])->name("send");
        Route::post("/{thread_id}/mark-read", [DealsMessagesController::class, "markAsRead"])->name("mark-read");
        Route::get("/unread-count", [DealsMessagesController::class, "getUnreadCount"])->name("unread-count");
    });
Route::get('/jobs', [JobController::class, 'list']);
Route::get('/check-subscription/{user_id}', [JobController::class, 'checkSubscription']);
Route::post('/apply-job', [JobController::class, 'apply']);
Route::get('/jobs/{job}/has-applied', [JobController::class, 'hasApplied']);


    //Zoom live class
    Route::get('zoom/settings', [ApiController::class, 'zoom_settings']);
    Route::get('zoom/meetings', [ApiController::class, 'live_class_schedules']);

    Route::get('payment/{token}', [ApiController::class, 'payment']);
    Route::get('token', [ApiController::class, 'token']);

    Route::get('free_course_enroll/{course_id}', [ApiController::class, 'free_course_enroll']);

    Route::get('cart_tools', [ApiController::class, 'cart_tools']);


    // مسارات نظام المحادثات البديل (للتوافق مع النظام السابق إذا لزم الأمر)
    Route::middleware(["auth:sanctum"])->group(function () {
        Route::get("/conversations", [ConversationController::class, "index"])->name("conversations.index");
        Route::get("/conversations/{conversation}", [ConversationController::class, "show"])->name("conversations.show");
        Route::post("/conversations/start", [ConversationController::class, "startConversation"])->name("conversations.start");

    });
});






// Screen Capture and Ban System API Routes
Route::middleware(["auth:sanctum"])->group(function () {
    // Report a screen capture violation
    Route::post("/report-screen-capture-violation", [App\Http\Controllers\Api\BanController::class, "reportViolation"]);

    // Check user ban status
    Route::get("/check-ban-status/{user_id}", [App\Http\Controllers\Api\BanController::class, "checkBanStatus"]);
});


