@extends('layouts.instructor')
@push('title', get_phrase('محادثة'))

@section('content')
<div class="container-fluid px-4 py-6">
    <div class="row justify-content-center">
        <div class="col-12 col-lg-8">
            <!-- Chat Header -->
            <div class="card border-0 shadow-sm mb-3">
                <div class="card-body py-3">
                    <div class="d-flex align-items-center justify-content-between">
                        <div class="d-flex align-items-center">
                            <!-- Back Button -->
                            <a href="{{ route('instructor.conversations.index') }}" 
                               class="btn btn-outline-secondary btn-sm me-3">
                                <i class="fas fa-arrow-right"></i>
                            </a>
                            
                            <!-- User Info -->
                            <div class="d-flex align-items-center">
                                @if($otherUser && $otherUser->photo)
                                    <img src="{{ asset($otherUser->photo) }}" 
                                         alt="{{ $otherUser->name }}" 
                                         class="rounded-circle me-3" 
                                         width="40" height="40"
                                         style="object-fit: cover;">
                                @else
                                    <div class="rounded-circle bg-primary d-flex align-items-center justify-content-center text-white fw-bold me-3" 
                                         style="width: 40px; height: 40px; font-size: 16px;">
                                        {{ $otherUser ? strtoupper(substr($otherUser->name, 0, 1)) : '؟' }}
                                    </div>
                                @endif
                                
                                <div>
                                    <h6 class="mb-0 fw-semibold">
                                        {{ $otherUser ? $otherUser->name : 'مستخدم غير معروف' }}
                                    </h6>
                                    <small class="text-muted">
                                        {{ $otherUser ? $otherUser->email : '' }}
                                    </small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Chat Options -->
                        <div class="dropdown">
                            <button class="btn btn-outline-secondary btn-sm" type="button" data-bs-toggle="dropdown">
                                <i class="fas fa-ellipsis-v"></i>
                            </button>
                            <ul class="dropdown-menu">
                                <li>
                                    <a class="dropdown-item" href="#" onclick="markAllAsRead()">
                                        <i class="fas fa-check-double me-2"></i>تحديد الكل كمقروء
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="#" onclick="reportConversation()">
                                        <i class="fas fa-flag me-2"></i>إبلاغ عن المحادثة
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Messages Container -->
            <div class="card border-0 shadow-sm">
                <div class="card-body p-0">
                    <div id="messages-container" class="p-4" style="height: 500px; overflow-y: auto;">
                        @foreach ($messages as $message)
                            <div class="d-flex {{ $message->sender_id == auth()->id() ? 'justify-content-end' : 'justify-content-start' }} mb-3" 
                                 data-message-id="{{ $message->id }}">
                                <div class="message-bubble {{ $message->sender_id == auth()->id() ? 'sent' : 'received' }}">
                                    <div class="message-content">
                                        {{ $message->message }}
                                    </div>
                                    <div class="message-meta">
                                        <small class="time">
                                            {{ $message->created_at->format("H:i") }}
                                        </small>
                                        @if ($message->sender_id == auth()->id())
                                            <small class="read-status ms-1">
                                                @if ($message->read)
                                                    <i class="fas fa-check-double text-primary" title="تمت القراءة"></i>
                                                @else
                                                    <i class="fas fa-check text-muted" title="تم الإرسال"></i>
                                                @endif
                                            </small>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
                
                <!-- Message Input -->
                <div class="card-footer bg-light border-0">
                    <form id="message-form" class="d-flex align-items-center" method="POST" action="{{ route("instructor.conversations.send-message") }}">

                        @csrf
                        <input type="hidden" name="thread_id" value="{{ $thread_id }}">
                        
                        <div class="flex-grow-1 me-3">
                            <input type="text" 
                                   name="message" 
                                   id="message-input" 
                                   class="form-control border-0 bg-white shadow-sm" 
                                   placeholder="اكتب رسالتك هنا..." 
                                   required
                                   style="border-radius: 25px; padding: 12px 20px;">
                        </div>
                        
                        <button type="submit" 
                                class="btn btn-primary rounded-circle d-flex align-items-center justify-content-center"
                                style="width: 45px; height: 45px;"
                                id="send-button">
                            <i class="fas fa-paper-plane"></i>
                        </button>
                    </form>
                    <div id="closed-conversation-message" class="alert alert-warning mt-3" style="display: none;">
                        <i class="fas fa-info-circle me-2"></i>
                        تم إنهاء هذه المحادثة من قبل فريق دعم مرسوم.
                        <span id="closure-reason-display" class="d-block mt-2"></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push("styles")
<style>
    .message-bubble {
        max-width: 70%;
        padding: 12px 16px;
        border-radius: 18px;
        position: relative;
        word-wrap: break-word;
    }
    
    .message-bubble.sent {
        background: linear-gradient(135deg, #0d6efd 0%, #0056b3 100%);
        color: white;
        margin-left: auto;
    }
    
    .message-bubble.received {
        background: #f8f9fa;
        color: #212529;
        border: 1px solid #e9ecef;
    }
    
    .message-content {
        font-size: 14px;
        line-height: 1.4;
        margin-bottom: 4px;
    }
    
    .message-meta {
        display: flex;
        align-items: center;
        justify-content: flex-end;
        font-size: 11px;
    }
    
    .message-bubble.sent .message-meta {
        color: rgba(255, 255, 255, 0.8);
    }
    
    .message-bubble.received .message-meta {
        color: #6c757d;
    }
    
    #messages-container {
        background: linear-gradient(to bottom, #f8f9fa 0%, #ffffff 100%);
    }
    
    #messages-container::-webkit-scrollbar {
        width: 6px;
    }
    
    #messages-container::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 3px;
    }
    
    #messages-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 3px;
    }
    
    #messages-container::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }
    
    .card {
        border-radius: 12px;
    }
    
    .card-footer {
        border-radius: 0 0 12px 12px !important;
    }
    
    #message-input:focus {
        box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
        border-color: #0d6efd;
    }
    
    .btn-primary {
        background: linear-gradient(135deg, #0d6efd 0%, #0056b3 100%);
        border: none;
        transition: all 0.2s ease-in-out;
    }
    
    .btn-primary:hover {
        transform: scale(1.05);
        box-shadow: 0 4px 12px rgba(13, 110, 253, 0.3);
    }
    
    .typing-indicator {
        display: none;
        padding: 12px 16px;
        background: #f8f9fa;
        border-radius: 18px;
        max-width: 70px;
        margin-bottom: 15px;
    }
    
    .typing-indicator span {
        height: 8px;
        width: 8px;
        background: #6c757d;
        border-radius: 50%;
        display: inline-block;
        margin-right: 3px;
        animation: typing 1.4s infinite ease-in-out;
    }
    
    .typing-indicator span:nth-child(2) {
        animation-delay: 0.2s;
    }
    
    .typing-indicator span:nth-child(3) {
        animation-delay: 0.4s;
    }
    
    @keyframes typing {
        0%, 60%, 100% {
            transform: translateY(0);
            opacity: 0.5;
        }
        30% {
            transform: translateY(-10px);
            opacity: 1;
        }
    }
    
    .message-bubble {
        animation: slideIn 0.3s ease-out;
    }
    
    @keyframes slideIn {
        from {
            opacity: 0;
            transform: translateY(10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
</style>
@endpush


@push("js")
<script>
    const messagesContainer = document.getElementById("messages-container");
    const messageForm = document.getElementById("message-form");
    const messageInput = document.getElementById("message-input");
    const sendButton = document.getElementById("send-button");
    const closedMessageDiv = document.getElementById("closed-conversation-message");
    const closureReasonDisplay = document.getElementById("closure-reason-display");
    const threadId = {{ $thread_id }};
    const currentUserId = {{ auth()->id() }};

    // Scroll to bottom on load
    function scrollToBottom() {
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
    scrollToBottom();

    // Check conversation status
    async function checkConversationStatus() {
        try {
            const response = await fetch(`/reports/check-conversation-status/${threadId}`);
            const data = await response.json();

            if (data.success && data.is_closed) {
                messageInput.disabled = true;
                sendButton.disabled = true;
                messageInput.placeholder = "لا يمكنك إرسال رسائل إلى هذه المحادثة.";
                closedMessageDiv.style.display = "block";
                if (data.closure_reason) {
                    closureReasonDisplay.textContent = `السبب: ${data.closure_reason}`;
                }
            } else {
                messageInput.disabled = false;
                sendButton.disabled = false;
                messageInput.placeholder = "اكتب رسالتك هنا...";
                closedMessageDiv.style.display = "none";
                closureReasonDisplay.textContent = "";
            }
        } catch (error) {
            console.error("Error checking conversation status:", error);
        }
    }

    checkConversationStatus();
    setInterval(checkConversationStatus, 10000);

    // Send message
    messageForm.addEventListener("submit", async function(e) {
        e.preventDefault();
        const messageText = messageInput.value.trim();
        if (!messageText) return;

        sendButton.disabled = true;
        sendButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';

        const formData = new FormData();
        formData.append("_token", document.querySelector('input[name="_token"]').value);
        formData.append('thread_id', threadId);
        formData.append('message', messageText);

        try {
            const response = await fetch('/instructor/conversations/send-message', {
                method: 'POST',
                body: formData,
                headers: { 'X-Requested-With': 'XMLHttpRequest' }
            });

            const data = await response.json();

            if (response.ok && data.success) {
                appendMessage(data.data);
                messageInput.value = "";
                scrollToBottom();
            } else {
                alert("فشل إرسال الرسالة: " + (data.message || "حدث خطأ."));
            }
        } finally {
            sendButton.disabled = false;
            sendButton.innerHTML = '<i class="fas fa-paper-plane"></i>';
        }
    });

    // Append new message
    function appendMessage(message) {
        const isSender = message.sender_id == currentUserId;
        const messageDiv = document.createElement("div");
        messageDiv.className = `d-flex ${isSender ? "justify-content-end" : "justify-content-start"} mb-3`;
        messageDiv.setAttribute('data-message-id', message.id);

        messageDiv.innerHTML = `
            <div class="message-bubble ${isSender ? 'sent' : 'received'}">
                <div class="message-content">
                    ${message.message}
                </div>
                <div class="message-meta">
                    <small class="time">
                        ${new Date(message.created_at).toLocaleTimeString([], { hour: "2-digit", minute: "2-digit" })}
                    </small>
                    ${isSender ? `
                        <small class="read-status ms-1">
                            <i class="fas fa-check text-muted" title="تم الإرسال"></i>
                        </small>` : ''}
                </div>
            </div>
        `;
        messagesContainer.appendChild(messageDiv);
    }

    // Mark all as read
    function markAllAsRead() {
        fetch(`/api/conversations/${threadId}/mark-read`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('input[name="_token"]').value,
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(() => {
            console.log('Messages marked as read');
        })
        .catch(error => console.error('Error marking messages as read:', error));
    }

    // Report conversation (تم التعديل هنا بالكامل)
    function reportConversation() {
        const confirmReport = confirm("هل أنت متأكد أنك تريد الإبلاغ عن هذه المحادثة؟ سيتم مراجعتها من قبل فريق الدعم.");
        if (!confirmReport) return;

        const csrfToken = document.querySelector('input[name="_token"]').value;

        fetch('/reports/store', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken,
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                conversation_id: threadId,
                reason: "محتوى غير لائق أو مخالف لسياسات الاستخدام."
            })
        })
        .then(async response => {
            const data = await response.json();
            if (response.ok && data.success) {
                alert(data.message || "تم إرسال الإبلاغ بنجاح.");
            } else {
                alert("فشل الإبلاغ: " + (data.message || `حدث خطأ (${response.status})`));
            }
        })
        .catch(error => {
            console.error("Error reporting conversation:", error);
            alert("حدث خطأ أثناء محاولة إرسال الإبلاغ.");
        });
    }

    // Poll for new messages
    setInterval(async () => {
        try {
            const response = await fetch(`/api/conversations/${threadId}/messages`, {
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const data = await response.json();

            if (response.ok && data.messages) {
                const currentMessageIds = Array.from(messagesContainer.children)
                    .map(div => div.getAttribute('data-message-id'))
                    .filter(id => id !== null);

                const newMessages = data.messages.filter(msg =>
                    !currentMessageIds.includes(msg.id.toString())
                );

                newMessages.forEach(msg => {
                    appendMessage(msg);
                });

                if (newMessages.length > 0) {
                    scrollToBottom();
                }
            }
        } catch (error) {
            console.error("Error fetching new messages:", error);
        }
    }, 3000);

    // Handle Enter key
    messageInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            messageForm.dispatchEvent(new Event('submit'));
        }
    });

    messageInput.focus();
</script>
@endpush
