@extends('layouts.admin')

@push('title', get_phrase('إدارة الإبلاغات'))

@section('content')
<div class="reports-management-container">
    <!-- Header Section -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-text">
                <h1 class="page-title">
                    <i class="fas fa-flag"></i>
                    إدارة الإبلاغات
                </h1>
                <p class="page-subtitle">إدارة ومراقبة الإبلاغات المرسلة من المستخدمين</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.reports.stats') }}" class="btn btn-info">
                    <i class="fas fa-chart-bar"></i>
                    الإحصائيات التفصيلية
                </a>
                <button onclick="exportReports()" class="btn btn-success">
                    <i class="fas fa-download"></i>
                    تصدير البيانات
                </button>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="stats-grid">
        <div class="stat-card pending">
            <div class="stat-icon">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-content">
                <h3 id="pending-count">{{ $reports->where('status', 'pending')->count() }}</h3>
                <p>في الانتظار</p>
            </div>
        </div>
        
        <div class="stat-card under-review">
            <div class="stat-icon">
                <i class="fas fa-search"></i>
            </div>
            <div class="stat-content">
                <h3 id="under-review-count">{{ $reports->where('status', 'under_review')->count() }}</h3>
                <p>قيد المراجعة</p>
            </div>
        </div>
        
        <div class="stat-card resolved">
            <div class="stat-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <h3 id="resolved-count">{{ $reports->where('status', 'resolved')->count() }}</h3>
                <p>تم الحل</p>
            </div>
        </div>
        
        <div class="stat-card total">
            <div class="stat-icon">
                <i class="fas fa-list"></i>
            </div>
            <div class="stat-content">
                <h3 id="total-count">{{ $reports->total() }}</h3>
                <p>إجمالي الإبلاغات</p>
            </div>
        </div>
    </div>

    <!-- Filters Section -->
    <div class="filters-container">
        <div class="filters-header">
            <h3>
                <i class="fas fa-filter"></i>
                تصفية النتائج
            </h3>
        </div>
        <form action="{{ route('admin.reports.index') }}" method="GET" class="filters-form">
            <div class="filters-grid">
                <div class="filter-group">
                    <label for="status">الحالة</label>
                    <select name="status" id="status" class="filter-select">
                        <option value="">جميع الحالات</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>في الانتظار</option>
                        <option value="under_review" {{ request('status') == 'under_review' ? 'selected' : '' }}>قيد المراجعة</option>
                        <option value="resolved" {{ request('status') == 'resolved' ? 'selected' : '' }}>تم الحل</option>
                        <option value="dismissed" {{ request('status') == 'dismissed' ? 'selected' : '' }}>تم الرفض</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="date_from">من تاريخ</label>
                    <input type="date" name="date_from" id="date_from" class="filter-input" value="{{ request('date_from') }}">
                </div>
                
                <div class="filter-group">
                    <label for="date_to">إلى تاريخ</label>
                    <input type="date" name="date_to" id="date_to" class="filter-input" value="{{ request('date_to') }}">
                </div>
                
                <div class="filter-group">
                    <label for="search">البحث</label>
                    <input type="text" name="search" id="search" class="filter-input" placeholder="البحث في السبب أو المحادثة..." value="{{ request('search') }}">
                </div>
            </div>
            
            <div class="filters-actions">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-search"></i>
                    تطبيق الفلاتر
                </button>
                <a href="{{ route('admin.reports.index') }}" class="btn btn-secondary">
                    <i class="fas fa-times"></i>
                    إعادة تعيين
                </a>
            </div>
        </form>
    </div>

    <!-- Main Table Container -->
    <div class="table-container">
        <div class="table-header">
            <h3>
                <i class="fas fa-list"></i>
                قائمة الإبلاغات
            </h3>
            <div class="table-actions">
                <div class="view-options">
                    <button class="view-btn active" data-view="table">
                        <i class="fas fa-table"></i>
                    </button>
                    <button class="view-btn" data-view="cards">
                        <i class="fas fa-th-large"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <div class="table-wrapper">
            <table class="reports-table">
                <thead>
                    <tr>
                        <th>
                            <i class="fas fa-hashtag"></i>
                            رقم الإبلاغ
                        </th>
                        <th>
                            <i class="fas fa-user"></i>
                            المبلغ
                        </th>
                        <th>
                            <i class="fas fa-comments"></i>
                            المحادثة
                        </th>
                        <th>
                            <i class="fas fa-exclamation-circle"></i>
                            سبب الإبلاغ
                        </th>
                        <th>
                            <i class="fas fa-toggle-on"></i>
                            الحالة
                        </th>
                        <th>
                            <i class="fas fa-user-cog"></i>
                            تمت المراجعة بواسطة
                        </th>
                        <th>
                            <i class="fas fa-calendar-plus"></i>
                            تاريخ الإبلاغ
                        </th>
                        <th>
                            <i class="fas fa-cogs"></i>
                            الإجراءات
                        </th>
                    </tr>
                </thead>
                <tbody>
                    @forelse ($reports as $report)
                        <tr class="report-row">
                            <td>
                                <div class="report-id">
                                    <span class="id-number">#{{ $report->id }}</span>
                                    <span class="conversation-id">محادثة: {{ $report->conversation_id }}</span>
                                </div>
                            </td>
                            <td>
                                <div class="user-info">
                                    @if ($report->reporter)
                                        <div class="user-avatar">
                                            @if($report->reporter->photo)
                                                <img src="{{ asset($report->reporter->photo) }}" alt="user-img" class="user-image">
                                            @else
                                                {{ substr($report->reporter->name, 0, 1) }}
                                            @endif
                                        </div>
                                        <div class="user-details">
                                            <span class="user-name">{{ $report->reporter->name }}</span>
                                            <span class="user-email">{{ $report->reporter->email }}</span>
                                        </div>
                                    @else
                                        <div class="user-info">
                                            <div class="user-avatar deleted">
                                                <i class="fas fa-user-slash"></i>
                                            </div>
                                            <div class="user-details">
                                                <span class="user-name deleted">مستخدم محذوف</span>
                                                <span class="user-email">غير متاح</span>
                                            </div>
                                        </div>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="conversation-info">
                                    <span class="conversation-id">#{{ $report->conversation_id }}</span>
                                    @if($report->conversation_closed)
                                        <span class="status-badge closed">
                                            <i class="fas fa-lock"></i>
                                            مغلقة
                                        </span>
                                    @else
                                        <span class="status-badge open">
                                            <i class="fas fa-unlock"></i>
                                            مفتوحة
                                        </span>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="report-reason">
                                    <span class="reason-text" title="{{ $report->reason }}">
                                        {{ Str::limit($report->reason, 50) }}
                                    </span>
                                    @if(strlen($report->reason) > 50)
                                        <button class="btn-expand" onclick="showFullReason('{{ $report->id }}', '{{ addslashes($report->reason) }}')">
                                            <i class="fas fa-expand-alt"></i>
                                        </button>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <span class="status-badge {{ $report->status }}">
                                    @switch($report->status)
                                        @case('pending')
                                            <i class="fas fa-clock"></i>
                                            في الانتظار
                                            @break
                                        @case('under_review')
                                            <i class="fas fa-search"></i>
                                            قيد المراجعة
                                            @break
                                        @case('resolved')
                                            <i class="fas fa-check-circle"></i>
                                            تم الحل
                                            @break
                                        @case('dismissed')
                                            <i class="fas fa-times-circle"></i>
                                            تم الرفض
                                            @break
                                    @endswitch
                                </span>
                            </td>
                            <td>
                                <div class="reviewer-info">
                                    @if($report->reviewer)
                                        <span class="reviewer-name">{{ $report->reviewer->name }}</span>
                                        <span class="review-date">{{ $report->reviewed_at ? $report->reviewed_at->format('d/m/Y H:i') : '' }}</span>
                                    @else
                                        <span class="reviewer-name unknown">لم تتم المراجعة</span>
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="created-date">
                                    <span class="date">{{ $report->created_at->format("d/m/Y") }}</span>
                                    <span class="time">{{ $report->created_at->format("H:i:s") }}</span>
                                </div>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <a href="{{ route('admin.reports.show', $report->id) }}" class="btn-action view" title="عرض التفاصيل">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    
                                    @if($report->status !== 'resolved' && $report->status !== 'dismissed')
                                        <button class="btn-action edit" title="تحديث الحالة" onclick="showStatusModal({{ $report->id }}, '{{ $report->status }}')">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    @endif
                                    
                                    @if(!$report->conversation_closed)
                                        <button class="btn-action close" title="إغلاق المحادثة" onclick="showCloseModal({{ $report->id }})">
                                            <i class="fas fa-lock"></i>
                                        </button>
                                    @endif
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="8" class="no-data">
                                <div class="no-data-content">
                                    <i class="fas fa-inbox"></i>
                                    <h4>لا توجد إبلاغات</h4>
                                    <p>لم يتم استلام أي إبلاغات حتى الآن</p>
                                </div>
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        
        @if($reports->hasPages())
        <div class="pagination-wrapper">
            {{ $reports->appends(request()->query())->links() }}
        </div>
        @endif
    </div>
</div>

<!-- Status Update Modal -->
<div id="statusModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>تحديث حالة الإبلاغ</h3>
            <span class="close" onclick="closeModal('statusModal')">&times;</span>
        </div>
        <form id="statusForm" method="POST">
            @csrf
            @method('PATCH')
            <div class="modal-body">
                <div class="form-group">
                    <label for="modal_status">الحالة الجديدة</label>
                    <select name="status" id="modal_status" class="form-control" required>
                        <option value="pending">في الانتظار</option>
                        <option value="under_review">قيد المراجعة</option>
                        <option value="resolved">تم الحل</option>
                        <option value="dismissed">تم الرفض</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="modal_admin_notes">ملاحظات الإدارة</label>
                    <textarea name="admin_notes" id="modal_admin_notes" class="form-control" rows="4" placeholder="أضف ملاحظاتك..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('statusModal')">إلغاء</button>
                <button type="submit" class="btn btn-primary">تحديث</button>
            </div>
        </form>
    </div>
</div>

<!-- Close Conversation Modal -->
<div id="closeModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>إغلاق المحادثة</h3>
            <span class="close" onclick="closeModal('closeModal')">&times;</span>
        </div>
        <form id="closeForm" method="POST">
            @csrf
            <div class="modal-body">
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>تحذير:</strong> إغلاق المحادثة سيمنع المستخدمين من إرسال رسائل جديدة نهائياً.
                </div>
                <div class="form-group">
                    <label for="modal_closure_reason">سبب الإغلاق *</label>
                    <textarea name="closure_reason" id="modal_closure_reason" class="form-control" rows="4" placeholder="اذكر سبب إغلاق المحادثة..." required></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal('closeModal')">إلغاء</button>
                <button type="submit" class="btn btn-danger">إغلاق المحادثة</button>
            </div>
        </form>
    </div>
</div>

<!-- Reason Modal -->
<div id="reasonModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>سبب الإبلاغ كاملاً</h3>
            <span class="close" onclick="closeModal('reasonModal')">&times;</span>
        </div>
        <div class="modal-body">
            <div id="fullReasonText" class="reason-full-text"></div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeModal('reasonModal')">إغلاق</button>
        </div>
    </div>
</div>

<style>
/* Main Container Styles */
.reports-management-container {
    padding: 2rem;
    background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
    min-height: 100vh;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

/* Page Header */
.page-header {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.page-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: #2d3748;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.page-title i {
    color: #667eea;
    font-size: 2rem;
}

.page-subtitle {
    color: #718096;
    font-size: 1.1rem;
    margin: 0.5rem 0 0 0;
}

.header-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

/* Buttons */
.btn {
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    border: none;
    font-weight: 600;
    font-size: 0.95rem;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    text-decoration: none;
}

.btn-info {
    background: linear-gradient(135deg, #4299e1 0%, #3182ce 100%);
    color: white;
}

.btn-info:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(66, 153, 225, 0.4);
}

.btn-success {
    background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
    color: white;
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(72, 187, 120, 0.4);
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-secondary {
    background: linear-gradient(135deg, #a0aec0 0%, #718096 100%);
    color: white;
}

.btn-danger {
    background: linear-gradient(135deg, #f56565 0%, #e53e3e 100%);
    color: white;
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    transition: all 0.3s ease;
    border-left: 4px solid;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 40px rgba(0,0,0,0.15);
}

.stat-card.pending {
    border-left-color: #ed8936;
}

.stat-card.under-review {
    border-left-color: #4299e1;
}

.stat-card.resolved {
    border-left-color: #48bb78;
}

.stat-card.total {
    border-left-color: #667eea;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}

.stat-card.pending .stat-icon {
    background: linear-gradient(135deg, #ed8936, #dd6b20);
    color: white;
}

.stat-card.under-review .stat-icon {
    background: linear-gradient(135deg, #4299e1, #3182ce);
    color: white;
}

.stat-card.resolved .stat-icon {
    background: linear-gradient(135deg, #48bb78, #38a169);
    color: white;
}

.stat-card.total .stat-icon {
    background: linear-gradient(135deg, #667eea, #764ba2);
    color: white;
}

.stat-content h3 {
    font-size: 2rem;
    font-weight: 700;
    color: #2d3748;
    margin: 0;
}

.stat-content p {
    color: #718096;
    margin: 0;
    font-size: 0.9rem;
}

/* Filters Container */
.filters-container {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    margin-bottom: 2rem;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.filters-header h3 {
    color: #2d3748;
    margin: 0 0 1rem 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 1rem;
}

.filter-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #4a5568;
}

.filter-select,
.filter-input {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e2e8f0;
    border-radius: 8px;
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.filter-select:focus,
.filter-input:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.filters-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
}

/* Table Container */
.table-container {
    background: white;
    border-radius: 16px;
    overflow: hidden;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.table-header {
    padding: 1.5rem;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.table-header h3 {
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.view-options {
    display: flex;
    gap: 0.5rem;
}

.view-btn {
    padding: 0.5rem;
    background: rgba(255,255,255,0.2);
    border: none;
    border-radius: 6px;
    color: white;
    cursor: pointer;
    transition: all 0.3s ease;
}

.view-btn.active,
.view-btn:hover {
    background: rgba(255,255,255,0.3);
}

.table-wrapper {
    overflow-x: auto;
}

.reports-table {
    width: 100%;
    border-collapse: collapse;
}

.reports-table th {
    background: #f7fafc;
    padding: 1rem;
    text-align: right;
    font-weight: 600;
    color: #4a5568;
    border-bottom: 2px solid #e2e8f0;
}

.reports-table th i {
    margin-left: 0.5rem;
    color: #667eea;
}

.reports-table td {
    padding: 1rem;
    border-bottom: 1px solid #e2e8f0;
    vertical-align: middle;
}

.report-row:hover {
    background-color: #f7fafc;
}

/* User Info Styles */
.user-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.user-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea, #764ba2);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    overflow: hidden;
}

.user-avatar.deleted {
    background: #a0aec0;
}

.user-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.user-details {
    display: flex;
    flex-direction: column;
}

.user-name {
    font-weight: 600;
    color: #2d3748;
}

.user-name.deleted {
    color: #a0aec0;
}

.user-email {
    font-size: 0.8rem;
    color: #718096;
}

/* Report ID Styles */
.report-id {
    display: flex;
    flex-direction: column;
}

.id-number {
    font-weight: 700;
    color: #667eea;
    font-size: 1.1rem;
}

.conversation-id {
    font-size: 0.8rem;
    color: #718096;
}

/* Conversation Info */
.conversation-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

/* Report Reason */
.report-reason {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.reason-text {
    flex: 1;
}

.btn-expand {
    background: none;
    border: none;
    color: #667eea;
    cursor: pointer;
    padding: 0.25rem;
    border-radius: 4px;
    transition: all 0.3s ease;
}

.btn-expand:hover {
    background: #f7fafc;
}

/* Status Badges */
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-badge.pending {
    background: #fed7d7;
    color: #c53030;
}

.status-badge.under_review {
    background: #bee3f8;
    color: #2b6cb0;
}

.status-badge.resolved {
    background: #c6f6d5;
    color: #276749;
}

.status-badge.dismissed {
    background: #e2e8f0;
    color: #4a5568;
}

.status-badge.open {
    background: #c6f6d5;
    color: #276749;
}

.status-badge.closed {
    background: #fed7d7;
    color: #c53030;
}

/* Reviewer Info */
.reviewer-info {
    display: flex;
    flex-direction: column;
}

.reviewer-name {
    font-weight: 600;
    color: #2d3748;
}

.reviewer-name.unknown {
    color: #a0aec0;
    font-style: italic;
}

.review-date {
    font-size: 0.8rem;
    color: #718096;
}

/* Created Date */
.created-date {
    display: flex;
    flex-direction: column;
}

.date {
    font-weight: 600;
    color: #2d3748;
}

.time {
    font-size: 0.8rem;
    color: #718096;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 0.5rem;
}

.btn-action {
    width: 35px;
    height: 35px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-action.view {
    background: #4299e1;
    color: white;
}

.btn-action.view:hover {
    background: #3182ce;
    transform: translateY(-2px);
}

.btn-action.edit {
    background: #ed8936;
    color: white;
}

.btn-action.edit:hover {
    background: #dd6b20;
    transform: translateY(-2px);
}

.btn-action.close {
    background: #f56565;
    color: white;
}

.btn-action.close:hover {
    background: #e53e3e;
    transform: translateY(-2px);
}

/* No Data */
.no-data {
    text-align: center;
    padding: 3rem;
}

.no-data-content i {
    font-size: 3rem;
    color: #a0aec0;
    margin-bottom: 1rem;
}

.no-data-content h4 {
    color: #4a5568;
    margin-bottom: 0.5rem;
}

.no-data-content p {
    color: #718096;
}

/* Pagination */
.pagination-wrapper {
    padding: 1.5rem;
    display: flex;
    justify-content: center;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: white;
    margin: 5% auto;
    border-radius: 16px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 20px 60px rgba(0,0,0,0.3);
}

.modal-header {
    padding: 1.5rem;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h3 {
    margin: 0;
    color: #2d3748;
}

.close {
    font-size: 1.5rem;
    font-weight: bold;
    cursor: pointer;
    color: #a0aec0;
}

.close:hover {
    color: #4a5568;
}

.modal-body {
    padding: 1.5rem;
}

.modal-footer {
    padding: 1.5rem;
    border-top: 1px solid #e2e8f0;
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #4a5568;
}

.form-control {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e2e8f0;
    border-radius: 8px;
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.form-control:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.alert-warning {
    background: #fef5e7;
    color: #744210;
    border: 1px solid #f6e05e;
}

.reason-full-text {
    background: #f7fafc;
    padding: 1rem;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    line-height: 1.6;
}

/* Responsive Design */
@media (max-width: 768px) {
    .reports-management-container {
        padding: 1rem;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .filters-grid {
        grid-template-columns: 1fr;
    }
    
    .table-wrapper {
        font-size: 0.8rem;
    }
    
    .action-buttons {
        flex-direction: column;
    }
}
</style>

<script>
// Modal Functions
function showStatusModal(reportId, currentStatus) {
    document.getElementById('statusForm').action = `/admin/reports/${reportId}/status`;
    document.getElementById('modal_status').value = currentStatus;
    document.getElementById('statusModal').style.display = 'block';
}

function showCloseModal(reportId) {
    document.getElementById('closeForm').action = `/admin/reports/${reportId}/close-conversation`;
    document.getElementById('closeModal').style.display = 'block';
}

function showFullReason(reportId, reason) {
    document.getElementById('fullReasonText').textContent = reason;
    document.getElementById('reasonModal').style.display = 'block';
}

function closeModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        if (event.target === modal) {
            modal.style.display = 'none';
        }
    });
}

// Export function
function exportReports() {
    // Implement export functionality
    alert('سيتم تنفيذ وظيفة التصدير قريباً');
}

// Auto-refresh statistics
setInterval(function() {
    fetch('/admin/api/reports/stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('pending-count').textContent = data.data.pending_reports;
                document.getElementById('under-review-count').textContent = data.data.under_review_reports;
                document.getElementById('resolved-count').textContent = data.data.resolved_reports;
                document.getElementById('total-count').textContent = data.data.total_reports;
            }
        })
        .catch(error => console.error('Error updating stats:', error));
}, 30000); // كل 30 ثانية
</script>
@endsection

