@extends('layouts.admin')

@push('title', get_phrase('إضافة اشتراك جديد'))


@section('content')
<div class="subscription-create-container">
    <!-- Header Section -->
    <div class="page-header">
        <div class="header-content">
            <div class="header-text">
                <h1 class="page-title">
                    <i class="fas fa-plus-circle"></i>
                    إضافة اشتراك جديد
                </h1>
                <p class="page-subtitle">تفعيل أو تجديد اشتراك المستخدم السنوي</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.jobs.subscriptions.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-right"></i>
                    العودة للقائمة
                </a>
            </div>
        </div>
    </div>

    <!-- Main Form -->
    <div class="form-container">
        <form id="subscriptionForm" class="subscription-form">
            @csrf
            
            <!-- User Selection Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3>
                        <i class="fas fa-user"></i>
                        اختيار المستخدم
                    </h3>
                    <p>ابحث عن المستخدم بالاسم أو البريد الإلكتروني</p>
                </div>
                
                <div class="form-group">
                    <label for="userSearch">البحث عن المستخدم</label>
                    <div class="search-input-container">
                        <input type="text" 
                               id="userSearch" 
                               placeholder="اكتب اسم المستخدم أو البريد الإلكتروني..."
                               autocomplete="off">
                        <div class="search-icon">
                            <i class="fas fa-search"></i>
                        </div>
                        <div class="loading-spinner" id="searchSpinner" style="display: none;">
                            <i class="fas fa-spinner fa-spin"></i>
                        </div>
                    </div>
                    <div id="searchResults" class="search-results"></div>
                </div>

                <!-- Selected User Display -->
                <div id="selectedUserCard" class="selected-user-card" style="display: none;">
                    <div class="user-card-header">
                        <h4>المستخدم المحدد</h4>
                        <button type="button" class="btn-clear" onclick="clearSelectedUser()">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="user-card-content">
                        <div class="user-avatar">
                            <span id="userInitial"></span>
                        </div>
                        <div class="user-details">
                            <h5 id="selectedUserName"></h5>
                            <p id="selectedUserEmail"></p>
                            <div class="current-subscription" id="currentSubscription"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Subscription Details Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3>
                        <i class="fas fa-calendar-alt"></i>
                        تفاصيل الاشتراك
                    </h3>
                    <p>حدد نوع الاشتراك وتاريخ الانتهاء</p>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="subscriptionType">نوع العملية</label>
                        <select id="subscriptionType" name="subscription_type" required>
                            <option value="">اختر نوع العملية</option>
                            <option value="new">اشتراك جديد</option>
                            <option value="extend">تمديد الاشتراك الحالي</option>
                            <option value="renew">تجديد اشتراك منتهي</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="expiresAt">تاريخ انتهاء الاشتراك</label>
                        <input type="date" 
                               id="expiresAt" 
                               name="expires_at" 
                               required
                               min="{{ date('Y-m-d', strtotime('+1 day')) }}">
                    </div>
                </div>

                <!-- Quick Date Selection -->
                <div class="quick-dates">
                    <label>اختيار سريع:</label>
                    <div class="quick-date-buttons">
                        <button type="button" class="btn-quick-date" onclick="setQuickDate(30)">
                            شهر واحد
                        </button>
                        <button type="button" class="btn-quick-date" onclick="setQuickDate(90)">
                            3 أشهر
                        </button>
                        <button type="button" class="btn-quick-date" onclick="setQuickDate(180)">
                            6 أشهر
                        </button>
                        <button type="button" class="btn-quick-date" onclick="setQuickDate(365)">
                            سنة كاملة
                        </button>
                    </div>
                </div>
            </div>

            <!-- Summary Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3>
                        <i class="fas fa-clipboard-check"></i>
                        ملخص العملية
                    </h3>
                </div>

                <div class="summary-card" id="summaryCard" style="display: none;">
                    <div class="summary-row">
                        <span class="summary-label">المستخدم:</span>
                        <span class="summary-value" id="summaryUserName">-</span>
                    </div>
                    <div class="summary-row">
                        <span class="summary-label">البريد الإلكتروني:</span>
                        <span class="summary-value" id="summaryUserEmail">-</span>
                    </div>
                    <div class="summary-row">
                        <span class="summary-label">نوع العملية:</span>
                        <span class="summary-value" id="summarySubscriptionType">-</span>
                    </div>
                    <div class="summary-row">
                        <span class="summary-label">تاريخ الانتهاء الجديد:</span>
                        <span class="summary-value" id="summaryExpiresAt">-</span>
                    </div>
                    <div class="summary-row">
                        <span class="summary-label">مدة الاشتراك:</span>
                        <span class="summary-value" id="summaryDuration">-</span>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                    <i class="fas fa-check"></i>
                    تفعيل الاشتراك
                </button>
                <button type="button" class="btn btn-secondary" onclick="resetForm()">
                    <i class="fas fa-redo"></i>
                    إعادة تعيين
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="modal">
    <div class="modal-content success">
        <div class="modal-header">
            <div class="success-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3>تم بنجاح!</h3>
        </div>
        <div class="modal-body" id="successMessage">
            <!-- Success message will be inserted here -->
        </div>
        <div class="modal-actions">
            <button class="btn btn-primary" onclick="closeSuccessModal()">
                حسناً
            </button>
            <button class="btn btn-secondary" onclick="addAnother()">
                إضافة اشتراك آخر
            </button>
        </div>
    </div>
</div>

<style>
/* Main Container */
.subscription-create-container {
    padding: 2rem;
    background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
    min-height: 100vh;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

/* Page Header */
.page-header {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.page-title {
    font-size: 2.5rem;
    font-weight: 700;
    color: #2d3748;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.page-title i {
    color: #48bb78;
    font-size: 2rem;
}

.page-subtitle {
    color: #718096;
    font-size: 1.1rem;
    margin: 0.5rem 0 0 0;
}

/* Form Container */
.form-container {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
}

.subscription-form {
    max-width: 800px;
    margin: 0 auto;
}

/* Form Sections */
.form-section {
    margin-bottom: 3rem;
    padding-bottom: 2rem;
    border-bottom: 1px solid #e2e8f0;
}

.form-section:last-child {
    border-bottom: none;
    margin-bottom: 0;
}

.section-header {
    margin-bottom: 2rem;
}

.section-header h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #2d3748;
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.section-header h3 i {
    color: #667eea;
}

.section-header p {
    color: #718096;
    margin: 0;
    font-size: 0.95rem;
}

/* Form Groups */
.form-group {
    margin-bottom: 1.5rem;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #4a5568;
    margin-bottom: 0.5rem;
    font-size: 0.95rem;
}

.form-group input,
.form-group select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1rem;
    transition: all 0.3s ease;
    background: white;
}

.form-group input:focus,
.form-group select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

/* Search Input */
.search-input-container {
    position: relative;
}

.search-icon {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #a0aec0;
    pointer-events: none;
}

.loading-spinner {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #667eea;
}

#userSearch {
    padding-left: 2.5rem;
}

/* Search Results */
.search-results {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 2px solid #e2e8f0;
    border-top: none;
    border-radius: 0 0 12px 12px;
    max-height: 300px;
    overflow-y: auto;
    z-index: 100;
    display: none;
}

.search-result-item {
    padding: 1rem;
    cursor: pointer;
    border-bottom: 1px solid #f7fafc;
    transition: background-color 0.2s ease;
}

.search-result-item:hover {
    background-color: #f7fafc;
}

.search-result-item:last-child {
    border-bottom: none;
}

.result-user-info {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.result-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
}

.result-details h5 {
    margin: 0;
    color: #2d3748;
    font-weight: 600;
}

.result-details p {
    margin: 0;
    color: #718096;
    font-size: 0.9rem;
}

.result-subscription {
    margin-top: 0.25rem;
    font-size: 0.8rem;
}

.result-subscription.active {
    color: #22543d;
}

.result-subscription.expired {
    color: #742a2a;
}

/* Selected User Card */
.selected-user-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 16px;
    padding: 1.5rem;
    color: white;
    margin-top: 1rem;
}

.user-card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.user-card-header h4 {
    margin: 0;
    font-weight: 600;
}

.btn-clear {
    background: rgba(255,255,255,0.2);
    border: none;
    color: white;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: background-color 0.3s ease;
}

.btn-clear:hover {
    background: rgba(255,255,255,0.3);
}

.user-card-content {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.user-avatar {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    background: rgba(255,255,255,0.2);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    font-weight: 600;
}

.user-details h5 {
    margin: 0 0 0.25rem 0;
    font-size: 1.2rem;
    font-weight: 600;
}

.user-details p {
    margin: 0 0 0.5rem 0;
    opacity: 0.9;
}

.current-subscription {
    font-size: 0.9rem;
    opacity: 0.8;
}

/* Quick Dates */
.quick-dates {
    margin-top: 1rem;
}

.quick-dates label {
    display: block;
    margin-bottom: 0.75rem;
    font-weight: 600;
    color: #4a5568;
}

.quick-date-buttons {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.btn-quick-date {
    padding: 0.5rem 1rem;
    border: 2px solid #e2e8f0;
    background: white;
    border-radius: 8px;
    cursor: pointer;
    font-size: 0.9rem;
    font-weight: 500;
    color: #4a5568;
    transition: all 0.3s ease;
}

.btn-quick-date:hover {
    border-color: #667eea;
    color: #667eea;
    background: #f7fafc;
}

/* Summary Card */
.summary-card {
    background: #f7fafc;
    border-radius: 12px;
    padding: 1.5rem;
    border: 2px solid #e2e8f0;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.75rem 0;
    border-bottom: 1px solid #e2e8f0;
}

.summary-row:last-child {
    border-bottom: none;
}

.summary-label {
    font-weight: 600;
    color: #4a5568;
}

.summary-value {
    color: #2d3748;
    font-weight: 500;
}

/* Buttons */
.btn {
    padding: 0.75rem 1.5rem;
    border-radius: 12px;
    border: none;
    font-weight: 600;
    font-size: 0.95rem;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-primary:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
}

.btn-primary:disabled {
    background: #e2e8f0;
    color: #a0aec0;
    cursor: not-allowed;
}

.btn-secondary {
    background: #f7fafc;
    color: #4a5568;
    border: 2px solid #e2e8f0;
}

.btn-secondary:hover {
    background: #edf2f7;
    border-color: #cbd5e0;
}

/* Form Actions */
.form-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid #e2e8f0;
}

/* Success Modal */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal-content {
    background-color: white;
    margin: 10% auto;
    padding: 0;
    border-radius: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    width: 90%;
    max-width: 500px;
    animation: fadeIn 0.3s ease-out;
}

.modal-content.success .modal-header {
    background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
    color: white;
    padding: 1.5rem;
    border-top-left-radius: 20px;
    border-top-right-radius: 20px;
    text-align: center;
}

.modal-content.success .success-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
}

.modal-content.success h3 {
    margin: 0;
    font-size: 1.8rem;
}

.modal-body {
    padding: 1.5rem;
    text-align: center;
    color: #4a5568;
    font-size: 1.1rem;
}

.modal-actions {
    padding: 1.5rem;
    display: flex;
    justify-content: center;
    gap: 1rem;
    border-top: 1px solid #e2e8f0;
}

/* Animations */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-20px); }
    to { opacity: 1; transform: translateY(0); }
}

/* Responsive */
@media (max-width: 768px) {
    .form-row {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
    let selectedUser = null;

    // Function to fetch user data
    async function fetchUsers(query) {
        if (query.length < 2) {
            document.getElementById('searchResults').style.display = 'none';
            return;
        }

        document.getElementById('searchSpinner').style.display = 'inline-block';
        document.querySelector('#userSearch + .search-icon').style.display = 'none';

        try {
            const response = await fetch(`{{ route('admin.jobs.subscriptions.search-users') }}?query=${query}`);
            const data = await response.json();
            displaySearchResults(data);
        } catch (error) {
            console.error('Error fetching users:', error);
            document.getElementById('searchResults').innerHTML = '<div class="search-result-item">حدث خطأ أثناء البحث.</div>';
            document.getElementById('searchResults').style.display = 'block';
        } finally {
            document.getElementById('searchSpinner').style.display = 'none';
            document.querySelector('#userSearch + .search-icon').style.display = 'inline-block';
        }
    }

    // Function to display search results
    function displaySearchResults(users) {
        const searchResultsDiv = document.getElementById('searchResults');
        searchResultsDiv.innerHTML = '';

        if (users.length === 0) {
            searchResultsDiv.innerHTML = '<div class="search-result-item">لا توجد نتائج.</div>';
            searchResultsDiv.style.display = 'block';
            return;
        }

        users.forEach(user => {
            const userDiv = document.createElement('div');
            userDiv.classList.add('search-result-item');
            userDiv.dataset.userId = user.id;
            userDiv.dataset.userName = user.name;
            userDiv.dataset.userEmail = user.email;
            userDiv.dataset.userSubscriptionExpiresAt = user.job_subscription_expires_at || 'لا يوجد اشتراك';

            const initial = user.name.charAt(0).toUpperCase();
            const subscriptionStatus = user.job_subscription_expires_at ? 
                (new Date(user.job_subscription_expires_at) > new Date() ? 
                    `<span class="result-subscription active">نشط حتى: ${new Date(user.job_subscription_expires_at).toLocaleDateString('en-GB')}</span>` : 
                    `<span class="result-subscription expired">منتهي في: ${new Date(user.job_subscription_expires_at).toLocaleDateString('en-GB')}</span>`) :
                `<span class="result-subscription">لا يوجد اشتراك</span>`;

            userDiv.innerHTML = `
                <div class="result-user-info">
                    <div class="result-avatar">${initial}</div>
                    <div class="result-details">
                        <h5>${user.name}</h5>
                        <p>${user.email}</p>
                        <p>${subscriptionStatus}</p>
                    </div>
                </div>
            `;
            userDiv.addEventListener('click', () => selectUser(user));
            searchResultsDiv.appendChild(userDiv);
        });

        searchResultsDiv.style.display = 'block';
    }

    // Function to select a user
    function selectUser(user) {
        selectedUser = user;
        document.getElementById('userSearch').value = user.name;
        document.getElementById('searchResults').style.display = 'none';
        document.getElementById('selectedUserCard').style.display = 'block';
        document.getElementById('userInitial').textContent = user.name.charAt(0).toUpperCase();
        document.getElementById('selectedUserName').textContent = user.name;
        document.getElementById('selectedUserEmail').textContent = user.email;

        const currentSubscriptionText = user.job_subscription_expires_at ? 
            `الاشتراك الحالي ينتهي في: ${new Date(user.job_subscription_expires_at).toLocaleDateString('en-GB')}` : 
            'لا يوجد اشتراك حالي';
        document.getElementById('currentSubscription').textContent = currentSubscriptionText;

        validateForm();
        updateSummary();
    }

    // Function to clear selected user
    function clearSelectedUser() {
        selectedUser = null;
        document.getElementById('userSearch').value = '';
        document.getElementById('selectedUserCard').style.display = 'none';
        document.getElementById('summaryCard').style.display = 'none';
        validateForm();
    }

    // Function to set quick date
    function setQuickDate(days) {
        const today = new Date();
        today.setDate(today.getDate() + days);
        const yyyy = today.getFullYear();
        const mm = String(today.getMonth() + 1).padStart(2, '0'); // Months start at 0!
        const dd = String(today.getDate()).padStart(2, '0');
        document.getElementById('expiresAt').value = `${yyyy}-${mm}-${dd}`;
        updateSummary();
    }

    // Function to validate form
    function validateForm() {
        const subscriptionType = document.getElementById('subscriptionType').value;
        const expiresAt = document.getElementById('expiresAt').value;
        const submitBtn = document.getElementById('submitBtn');
        const isValid = selectedUser && subscriptionType && expiresAt;
        submitBtn.disabled = !isValid;
        updateSummary();
    }

    // Function to update summary
    function updateSummary() {
        if (!selectedUser) return;
        
        const subscriptionType = document.getElementById('subscriptionType').value;
        const expiresAt = document.getElementById('expiresAt').value;
        
        if (!subscriptionType || !expiresAt) return;
        
        const summaryCard = document.getElementById('summaryCard');
        const typeLabels = {
            'new': 'اشتراك جديد',
            'extend': 'تمديد الاشتراك',
            'renew': 'تجديد الاشتراك'
        };
        
        // Calculate duration
        const today = new Date();
        const expiry = new Date(expiresAt);
        const diffTime = Math.abs(expiry - today);
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        
        // Format date in Gregorian calendar (dd/mm/yyyy)
        const formattedDate = expiry.toLocaleDateString('en-GB'); // This will give dd/mm/yyyy format
        
        document.getElementById('summaryUserName').textContent = selectedUser.name;
        document.getElementById('summaryUserEmail').textContent = selectedUser.email;
        document.getElementById('summarySubscriptionType').textContent = typeLabels[subscriptionType] || subscriptionType;
        document.getElementById('summaryExpiresAt').textContent = formattedDate;
        document.getElementById('summaryDuration').textContent = `${diffDays} يوم`;
        
        summaryCard.style.display = 'block';
    }

    // Function to handle form submission
    function handleFormSubmit(e) {
        e.preventDefault();
        
        if (!selectedUser) {
            alert('يرجى اختيار مستخدم');
            return;
        }
        
        const formData = {
            user_id: selectedUser.id,
            subscription_type: document.getElementById('subscriptionType').value,
            expires_at: document.getElementById('expiresAt').value
        };
        
        console.log('Sending data:', formData); // للتصحيح
        
        const submitBtn = document.getElementById('submitBtn');
        const originalText = submitBtn.innerHTML;
        
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التفعيل...';
        submitBtn.disabled = true;
        
        fetch('{{ route("admin.jobs.subscriptions.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf_token"]').getAttribute('content')
            },
            body: JSON.stringify(formData)
        })
        .then(response => {
            console.log('Response status:', response.status); // للتصحيح
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Response data:', data); // للتصحيح
            if (data.success) {
                showSuccessModal(data);
            } else {
                alert(data.message || 'حدث خطأ أثناء تفعيل الاشتراك');
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('حدث خطأ أثناء تفعيل الاشتراك: ' + error.message);
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }

    // Function to show success modal
    function showSuccessModal(data) {
        const modal = document.getElementById('successModal');
        const messageContainer = document.getElementById('successMessage');
        
        messageContainer.innerHTML = `
            <p>تم تفعيل الاشتراك للمستخدم <strong>${data.user.name}</strong> بنجاح.</p>
            <p>تاريخ الانتهاء الجديد: <strong>${data.user.expires_at_formatted}</strong></p>
        `;
        modal.style.display = 'block';
    }

    // Function to close success modal
    function closeSuccessModal() {
        document.getElementById('successModal').style.display = 'none';
    }

    // Function to add another subscription
    function addAnother() {
        closeSuccessModal();
        resetForm();
    }

    // Function to reset form
    function resetForm() {
        document.getElementById('subscriptionForm').reset();
        clearSelectedUser();
        document.getElementById('summaryCard').style.display = 'none';
        document.getElementById('submitBtn').disabled = true;
    }

    // Event Listeners
    document.getElementById('userSearch').addEventListener('input', (e) => fetchUsers(e.target.value));
    document.getElementById('subscriptionType').addEventListener('change', validateForm);
    document.getElementById('expiresAt').addEventListener('change', validateForm);
    document.getElementById('subscriptionForm').addEventListener('submit', handleFormSubmit);

    // Close search results when clicking outside
    document.addEventListener('click', function(event) {
        const searchResults = document.getElementById('searchResults');
        const userSearch = document.getElementById('userSearch');
        if (!searchResults.contains(event.target) && !userSearch.contains(event.target)) {
            searchResults.style.display = 'none';
        }
    });

    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
        const modal = document.getElementById('successModal');
        if (event.target == modal) {
            closeSuccessModal();
        }
    });
</script>
@endsection
