<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;
use App\Models\Dispute;
use App\Models\WalletTransaction;
use App\Models\WithdrawalRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Composer للنزاعات (Disputes)
        View::composer('*', function ($view) {
            if (Auth::check() && Auth::user()->can('admin.disputes')) {
                $disputesCount = Cache::remember('disputes_counts', 300, function () {
                    return $this->getDisputesCount();
                });
                
                $view->with($disputesCount);
            } else {
                $view->with([
                    'pending_disputes_count' => 0,
                    'in_progress_disputes_count' => 0,
                    'resolved_disputes_count' => 0,
                    'total_disputes_count' => 0
                ]);
            }
        });

        // Composer للمعاملات المالية (Financial)
        View::composer('*', function ($view) {
            if (Auth::check() && Auth::user()->can('admin.financial')) {
                $financialCounts = Cache::remember('financial_counts', 300, function () {
                    return $this->getFinancialCounts();
                });
                
                $view->with($financialCounts);
            } else {
                $view->with([
                    'pendingTransactionsCount' => 0,
                    'pendingWithdrawalsCount' => 0
                ]);
            }
        });
    }

    /**
     * الحصول على إحصائيات النزاعات
     */
    protected function getDisputesCount(): array
    {
        return [
            'pending_disputes_count' => Dispute::where('status', 'pending')->count(),
            'in_progress_disputes_count' => Dispute::where('status', 'in_progress')->count(),
            'resolved_disputes_count' => Dispute::where('status', 'resolved')->count(),
            'total_disputes_count' => Dispute::count()
        ];
    }

    /**
     * الحصول على إحصائيات المعاملات المالية
     */
    protected function getFinancialCounts(): array
    {
        return [
            'pendingTransactionsCount' => WalletTransaction::where('status', 'pending')->count(),
            'pendingWithdrawalsCount' => WithdrawalRequest::where('status', 'pending')->count()
        ];
    }
}