<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable implements MustVerifyEmail

{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'status',
        'email_verified_at',
        'photo',
        'wallet_id',
        'wallet_balance',
        'frozen_balance',
        'job_subscription_expires_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'job_subscription_expires_at' => 'datetime',
    ];
protected $appends = ['total_balance'];

    // العلاقات
    public function sentTransactions()
    {
        return $this->hasMany(WalletTransaction::class, 'sender_id');
    }

    public function receivedTransactions()
    {
        return $this->hasMany(WalletTransaction::class, 'receiver_id');
    }

    public function withdrawalRequests()
    {
        return $this->hasMany(WithdrawalRequest::class);
    }

    // السمات المضافة
    public function getTotalBalanceAttribute()
    {
        return $this->wallet_balance + $this->frozen_balance;
    }

    // نطاقات الاستعلام
    public function scopeHasSufficientBalance($query, $amount)
    {
        return $query->where('wallet_balance', '>=', $amount);
    }

    // التوابع المساعدة
    public function canSendMoney($amount)
    {
        return $this->wallet_balance >= $amount;
    }

    public function hasPendingTransactions()
    {
        return $this->sentTransactions()->where('status', 'pending')->exists();
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($user) {
            if (empty($user->wallet_id)) {
                $user->wallet_id = self::generateWalletId();
            }
        });
    }


protected $dates = ['subscription_expires_at'];

    private static function generateWalletId()
    {
        $prefix = 'MRSOOM';
        $unique = strtoupper(uniqid());
        return "{$prefix}{$unique}";
    }

 public function conversationsAsUserOne()
    {
        return $this->hasMany(Conversation::class, 'user_one_id');
    }

    /**
     * Get conversations where this user is user_two.
     */
    public function conversationsAsUserTwo()
    {
        return $this->hasMany(Conversation::class, 'user_two_id');
    }

    /**
     * Get all conversations involving this user.
     */
    public function conversations()
    {
        return $this->hasMany(Conversation::class, 'user_one_id')
                    ->orWhere('user_two_id', $this->id);
    }

    public function activeBan()
    {
        return $this->hasOne(UserBan::class)->where("is_active", true)->where(function ($query) {
            $query->whereNull("banned_until")->orWhere("banned_until", ">", now());
        });
    }

    public function bans()
    {
        return $this->hasMany(UserBan::class);
    }

    public function screenCaptureViolations()
    {
        return $this->hasMany(ScreenCaptureViolation::class);
    }

    /**
     * Get messages sent by this user.
     */
    public function chatMessages()
    {
        return $this->hasMany(ChatMessages::class, 'sender_id');
    }

    /**
     * Find or create a conversation with another user.
     */
    public function getOrCreateConversationWith(User $otherUser)
    {
        // Check if a conversation already exists in either direction
        $conversation = $this->conversationsAsUserOne()->where('user_two_id', $otherUser->id)->first();
        if (!$conversation) {
            $conversation = $this->conversationsAsUserTwo()->where('user_one_id', $otherUser->id)->first();
        }

        // If no conversation exists, create a new one
        if (!$conversation) {
            $conversation = Conversation::create([
                'user_one_id' => $this->id,
                'user_two_id' => $otherUser->id,
            ]);
        }

        return $conversation;
    }
}
