<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Report extends Model
{
    use HasFactory;

    protected $fillable = [
        'conversation_id',
        'reporter_id',
        'reason',
        'status',
        'admin_notes',
        'reviewed_by',
        'reviewed_at',
        'conversation_closed'
    ];

    protected $casts = [
        'reviewed_at' => 'datetime',
        'conversation_closed' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * حالات الإبلاغ المتاحة
     */
    const STATUS_PENDING = 'pending';
    const STATUS_UNDER_REVIEW = 'under_review';
    const STATUS_RESOLVED = 'resolved';
    const STATUS_DISMISSED = 'dismissed';

    /**
     * الحصول على جميع حالات الإبلاغ
     */
    public static function getStatuses()
    {
        return [
            self::STATUS_PENDING => 'في الانتظار',
            self::STATUS_UNDER_REVIEW => 'قيد المراجعة',
            self::STATUS_RESOLVED => 'تم الحل',
            self::STATUS_DISMISSED => 'تم الرفض',
        ];
    }

    /**
     * الحصول على اسم الحالة باللغة العربية
     */
    public function getStatusNameAttribute()
    {
        return self::getStatuses()[$this->status] ?? 'غير معروف';
    }

    /**
     * العلاقة مع المستخدم الذي قام بالإبلاغ
     */
    public function reporter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reporter_id');
    }

    /**
     * العلاقة مع الإداري الذي راجع الإبلاغ
     */
    public function reviewer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * العلاقة مع المحادثة المغلقة
     */
    public function closedConversation(): HasOne
    {
        return $this->hasOne(ClosedConversation::class);
    }

    /**
     * التحقق من وجود إبلاغ نشط للمحادثة
     */
    public static function hasActiveReport($conversationId)
    {
        return self::where('conversation_id', $conversationId)
            ->whereIn('status', [self::STATUS_PENDING, self::STATUS_UNDER_REVIEW])
            ->exists();
    }

    /**
     * التحقق من وجود إبلاغ سابق من نفس المستخدم عن نفس المحادثة
     */
    public static function hasUserReported($conversationId, $userId)
    {
        return self::where('conversation_id', $conversationId)
            ->where('reporter_id', $userId)
            ->exists();
    }

    /**
     * التحقق من إمكانية الإبلاغ عن المحادثة
     */
    public static function canReportConversation($conversationId, $userId)
    {
        // التحقق من إغلاق المحادثة
        if (self::isConversationClosed($conversationId)) {
            return [
                'can_report' => false,
                'reason' => 'تم إنهاء هذه المحادثة من قبل فريق دعم مرسوم'
            ];
        }

        // التحقق من وجود إبلاغ سابق من نفس المستخدم
        if (self::hasUserReported($conversationId, $userId)) {
            return [
                'can_report' => false,
                'reason' => 'لقد قمت بالإبلاغ عن هذه المحادثة من قبل'
            ];
        }

        return [
            'can_report' => true,
            'reason' => null
        ];
    }

    /**
     * التحقق من إغلاق المحادثة
     */
    public static function isConversationClosed($conversationId)
    {
        return ClosedConversation::where('conversation_id', $conversationId)->exists();
    }

    /**
     * الحصول على الإبلاغات حسب الحالة
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * الحصول على الإبلاغات الحديثة
     */
    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * الحصول على الإبلاغات المراجعة
     */
    public function scopeReviewed($query)
    {
        return $query->whereNotNull('reviewed_by');
    }

    /**
     * الحصول على الإبلاغات غير المراجعة
     */
    public function scopeUnreviewed($query)
    {
        return $query->whereNull('reviewed_by');
    }

    /**
     * تعليم الإبلاغ كمراجع
     */
    public function markAsReviewed($adminId, $status = null, $notes = null)
    {
        $this->update([
            'reviewed_by' => $adminId,
            'reviewed_at' => now(),
            'status' => $status ?? $this->status,
            'admin_notes' => $notes ?? $this->admin_notes,
        ]);
    }

    /**
     * إغلاق المحادثة المرتبطة بالإبلاغ
     */
    public function closeConversation($adminId, $reason)
    {
        // إنشاء سجل في جدول المحادثات المغلقة
        ClosedConversation::create([
            'conversation_id' => $this->conversation_id,
            'closed_by' => $adminId,
            'closure_reason' => $reason,
            'report_id' => $this->id,
        ]);

        // تحديث حالة الإبلاغ
        $this->update([
            'conversation_closed' => true,
            'status' => self::STATUS_RESOLVED,
            'reviewed_by' => $adminId,
            'reviewed_at' => now(),
        ]);
    }
}

