<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class WithdrawalRequest extends Model
{
    use HasFactory;

    protected $fillable = [
        'request_id',
        'user_id',
        'amount',
        'withdrawal_method',
        'withdrawal_details',
        'status',
        'admin_notes',
        'processed_by',
        'processed_at',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'withdrawal_details' => 'array',
        'processed_at' => 'datetime',
    ];

    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    // العلاقات
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function admin()
    {
        return $this->belongsTo(User::class, 'processed_by');
    }

    // نطاقات الاستعلام
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    public function scopeRejected($query)
    {
        return $query->where('status', self::STATUS_REJECTED);
    }

    // التوابع المساعدة
    public function isPending()
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isApproved()
    {
        return $this->status === self::STATUS_APPROVED;
    }

    public function isRejected()
    {
        return $this->status === self::STATUS_REJECTED;
    }

    public function methodName()
    {
        return $this->withdrawal_method === 'bank_transfer' 
            ? 'حوالة بنكية' 
            : 'محفظة إلكترونية';
    }

    protected static function boot()
{
    parent::boot();

    static::creating(function ($request) {
        if (empty($request->request_id)) {
            $request->request_id = self::generateRequestId();
        }
        if (empty($request->status)) {
            $request->status = self::STATUS_PENDING; // تعيين الحالة الافتراضية
        }
    });
}


    private static function generateRequestId()
    {
        $prefix = 'WDR';
        $unique = strtoupper(uniqid());
        return "{$prefix}-{$unique}";
    }
}