<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\WalletTransaction;
use App\Models\WithdrawalRequest;

class WalletController extends Controller
{
    public function index()
    {
        $user = auth()->user();

        $availableBalance = $user->available_balance ?? $user->wallet_balance;
        $frozenBalance = $user->frozen_balance ?? 0;
        $totalBalance = $availableBalance + $frozenBalance;

        $transactions = WalletTransaction::where("sender_id", $user->wallet_id)
                            ->orWhere("receiver_id", $user->wallet_id)
                            ->orderBy("created_at", "desc")
                            ->get();

        $withdrawals = WithdrawalRequest::where('user_id', $user->id)
                            ->orderBy('created_at', 'desc')
                            ->get();

        return view('instructor.wallet.wallet', compact(
            'totalBalance', 'availableBalance', 'frozenBalance', 
            'transactions', 'withdrawals', 'user'
        ));
    }

    public function sendMoney(Request $request)
    {
        $request->validate([
            'wallet_id' => 'required|exists:users,wallet_id',
            'amount' => 'required|numeric|min:0.01',
        ]);

        $user = auth()->user();
        $recipient = User::where('wallet_id', $request->wallet_id)->first();

        if ($user->wallet_id === $request->wallet_id) {
            return response()->json(['error' => 'لا يمكن التحويل لنفس المحفظة']);
        }

        // Debugging: Log available_balance before check
        $userAvailableBalance = $user->available_balance ?? $user->wallet_balance;
        Log::info('User available_balance: ' . $userAvailableBalance . ' Requested amount: ' . $request->amount);
        if ($userAvailableBalance < $request->amount) {
            return response()->json(['error' => 'رصيد غير كافي']);
        }

        DB::beginTransaction();
        try {
            // تحديث الرصيد المتاح للمرسل
            if ($user->available_balance !== null) {
                $user->update([
                    'available_balance' => $user->available_balance - $request->amount
                ]);
            } else {
                $user->update([
                    'wallet_balance' => $user->wallet_balance - $request->amount
                ]);
            }

            // تحديث الرصيد المجمد للمستلم
            if ($recipient->frozen_balance !== null) {
                $recipient->update([
                    'frozen_balance' => $recipient->frozen_balance + $request->amount
                ]);
            } else {
                $recipient->update([
                    'frozen_balance' => $request->amount
                ]);
            }

            $transaction = WalletTransaction::create([
                'sender_id' => $user->id,
                'receiver_id' => $recipient->id,
                'amount' => $request->amount,
                'type' => WalletTransaction::TYPE_TRANSFER,
                'status' => WalletTransaction::STATUS_PENDING,
                'notes' => 'تحويل قيد الانتظار لـ ' . $recipient->name,
            ]);

            DB::commit();
            return response()->json([
                'success' => 'تم التحويل بنجاح بانتظار الموافقة',
                'transaction_id' => $transaction->id
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'حدث خطأ: '.$e->getMessage()]);
        }
    }

    public function withdrawMoney(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'withdrawal_method' => 'required|string',
            'withdrawal_details' => 'required|string',
        ]);

        $user = auth()->user();

        // Debugging: Log available_balance before check
        $userAvailableBalance = $user->available_balance ?? $user->wallet_balance;
        Log::info("Withdrawal - User available_balance: " . $userAvailableBalance . " Requested amount: " . $request->amount);
        if ($userAvailableBalance < $request->amount) {
            return response()->json(['error' => 'رصيد غير كافي']);
        }

        DB::beginTransaction();
        try {
            // تحديث الرصيد المتاح للمستخدم
            if ($user->available_balance !== null) {
                $user->update([
                    'available_balance' => $user->available_balance - $request->amount
                ]);
            } else {
                $user->update([
                    'wallet_balance' => $user->wallet_balance - $request->amount
                ]);
            }

            WithdrawalRequest::create([
                'user_id' => $user->id,
                'amount' => $request->amount,
                'withdrawal_method' => $request->withdrawal_method,
                'withdrawal_details' => $request->withdrawal_details,
                'status' => WithdrawalRequest::STATUS_PENDING,
            ]);

            WalletTransaction::create([
                'sender_id' => $user->id,
                'receiver_id' => null,
                'amount' => $request->amount,
                'type' => WalletTransaction::TYPE_WITHDRAW,
                'status' => WalletTransaction::STATUS_PENDING,
                'notes' => 'طلب سحب - ' . $request->withdrawal_method,
            ]);

            DB::commit();
            return response()->json(['success' => 'تم تقديم طلب السحب']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'حدث خطأ: '.$e->getMessage()]);
        }
    }

    public function confirmTransaction($transactionId)
    {
        DB::beginTransaction();
        try {
            $transaction = WalletTransaction::findOrFail($transactionId);

            if ($transaction->status !== WalletTransaction::STATUS_PENDING) {
                return back()->with('error', 'لا يمكن تعديل معاملة مكتملة أو ملغاة');
            }

            $receiver = User::where('wallet_id', $transaction->receiver_id)->first();

            if ($receiver) {
                $receiver->update([
                    'frozen_balance' => $receiver->frozen_balance - $transaction->amount,
                    'available_balance' => $receiver->available_balance + $transaction->amount,
                ]);
            }

            $transaction->update([
                'status' => WalletTransaction::STATUS_COMPLETED,
                'notes' => 'تمت الموافقة على التحويل',
            ]);

            DB::commit();
            return back()->with('success', 'تمت الموافقة على التحويل');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ: '.$e->getMessage());
        }
    }

    public function cancelTransaction($transactionId)
    {
        DB::beginTransaction();
        try {
            $transaction = WalletTransaction::findOrFail($transactionId);

            if ($transaction->status !== WalletTransaction::STATUS_PENDING) {
                return back()->with('error', 'لا يمكن تعديل معاملة مكتملة أو ملغاة');
            }

            $sender = User::where('wallet_id', $transaction->sender_id)->first();
            $receiver = User::where('wallet_id', $transaction->receiver_id)->first();

            if ($sender) {
                $sender->update([
                    'available_balance' => $sender->available_balance + $transaction->amount
                ]);
            }

            if ($receiver) {
                $receiver->update([
                    'frozen_balance' => $receiver->frozen_balance - $transaction->amount
                ]);
            }

            $transaction->update([
                'status' => WalletTransaction::STATUS_CANCELLED,
                'notes' => 'تم إلغاء التحويل',
            ]);

            DB::commit();
            return back()->with('success', 'تم إلغاء التحويل وإرجاع المبلغ');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ: '.$e->getMessage());
        }
    }
}

