<?php

namespace App\Http\Controllers;

use App\Models\Report;
use App\Models\ClosedConversation;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReportController extends Controller
{
    /**
     * عرض جميع الإبلاغات في لوحة التحكم
     */
    public function index(Request $request)
    {
        $query = Report::with(['reporter', 'reviewer'])
            ->orderBy('created_at', 'desc');

        // تطبيق الفلاتر
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reason', 'LIKE', "%{$search}%")
                  ->orWhere('admin_notes', 'LIKE', "%{$search}%")
                  ->orWhere('conversation_id', 'LIKE', "%{$search}%")
                  ->orWhereHas('reporter', function($userQuery) use ($search) {
                      $userQuery->where('name', 'LIKE', "%{$search}%")
                               ->orWhere('email', 'LIKE', "%{$search}%");
                  });
            });
        }

        $reports = $query->paginate(20);
        $statuses = Report::getStatuses();

        $pending_reports_count = Report::where('status', 'pending')->count();

        return view('admin.reports.index', compact('reports', 'statuses'));
    }

    /**
     * عرض تفاصيل إبلاغ واحد
     */
    public function show($id)
    {
        $report = Report::with(['reporter', 'reviewer', 'closedConversation'])
            ->findOrFail($id);

        // جلب رسائل المحادثة
        $conversationMessages = $this->getConversationMessages($report->conversation_id);

        return view('admin.reports.show', compact('report', 'conversationMessages'));
    }

    /**
     * عرض المحادثة المرتبطة بالإبلاغ
     */
    public function viewConversation($id)
    {
        $report = Report::findOrFail($id);
        
        // جلب رسائل المحادثة
        $conversationMessages = $this->getConversationMessages($report->conversation_id);
        
        return view('admin.reports.conversation', compact('report', 'conversationMessages'));
    }

    /**
     * جلب رسائل المحادثة من قاعدة البيانات
     */
    private function getConversationMessages($conversationId)
    {
        try {
            // استخدام جدول deals_messages فقط كما طلب المستخدم
            return DB::table('deals_messages')
                ->leftJoin('users as sender', 'deals_messages.sender_id', '=', 'sender.id')
                ->leftJoin('users as receiver', 'deals_messages.receiver_id', '=', 'receiver.id')
                ->where('deals_messages.thread_id', $conversationId)
                ->select(
                    'deals_messages.id',
                    'deals_messages.thread_id',
                    'deals_messages.sender_id',
                    'deals_messages.receiver_id',
                    'deals_messages.message',
                    'deals_messages.read',
                    'deals_messages.created_at',
                    'deals_messages.updated_at',
                    'sender.name as sender_name',
                    'sender.email as sender_email',
                    'receiver.name as receiver_name',
                    'receiver.email as receiver_email'
                )
                ->orderBy('deals_messages.created_at', 'asc')
                ->get();
            
        } catch (Exception $e) {
            Log::error('Error fetching conversation messages from deals_messages: ' . $e->getMessage());
            return collect();
        }
    }

    /**
     * تحديث حالة الإبلاغ
     */
    public function updateStatus(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:pending,under_review,resolved,dismissed',
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            $report = Report::findOrFail($id);
            
            $report->markAsReviewed(
                Auth::id(),
                $request->status,
                $request->admin_notes
            );

            DB::commit();

            return redirect()->back()
                ->with('success', 'تم تحديث حالة الإبلاغ بنجاح');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error updating report status: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تحديث حالة الإبلاغ');
        }
    }

    /**
     * إغلاق المحادثة
     */
    public function closeConversation(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'closure_reason' => 'required|string|max:1000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            $report = Report::findOrFail($id);

            // التحقق من عدم إغلاق المحادثة مسبقاً
            if (ClosedConversation::isConversationClosed($report->conversation_id)) {
                return redirect()->back()
                    ->with('error', 'هذه المحادثة مغلقة بالفعل');
            }

            $report->closeConversation(Auth::id(), $request->closure_reason);

            DB::commit();

            return redirect()->back()
                ->with('success', 'تم إغلاق المحادثة بنجاح');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error closing conversation: ' . $e->getMessage());
            
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إغلاق المحادثة');
        }
    }

    /**
     * API: إنشاء إبلاغ جديد
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'conversation_id' => 'required|integer',
            'reason' => 'required|string|max:1000|min:10',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'البيانات المدخلة غير صحيحة',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $userId = Auth::id();
            $conversationId = $request->conversation_id;

            // التحقق من إمكانية الإبلاغ
            $canReport = Report::canReportConversation($conversationId, $userId);
            
            if (!$canReport['can_report']) {
                return response()->json([
                    'success' => false,
                    'message' => $canReport['reason']
                ], 400);
            }

            DB::beginTransaction();

            $report = Report::create([
                'conversation_id' => $conversationId,
                'reporter_id' => $userId,
                'reason' => $request->reason,
                'status' => Report::STATUS_PENDING,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم إرسال الإبلاغ بنجاح. سيتم مراجعته من قبل فريق الدعم.',
                'data' => [
                    'report_id' => $report->id,
                    'status' => $report->status_name,
                ]
            ], 201);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error creating report: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إرسال الإبلاغ. يرجى المحاولة مرة أخرى.'
            ], 500);
        }
    }

    /**
     * API: التحقق من وجود إبلاغ نشط للمحادثة
     */
    public function checkActiveReport($conversationId): JsonResponse
    {
        try {
            $hasActiveReport = Report::hasActiveReport($conversationId);
            
            return response()->json([
                'success' => true,
                'has_active_report' => $hasActiveReport
            ]);

        } catch (Exception $e) {
            Log::error('Error checking active report: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التحقق من الإبلاغ'
            ], 500);
        }
    }

    /**
     * API: التحقق من إمكانية الإبلاغ عن المحادثة
     */
    public function canReport($conversationId): JsonResponse
    {
        try {
            $userId = Auth::id();
            $canReport = Report::canReportConversation($conversationId, $userId);
            
            return response()->json([
                'success' => true,
                'can_report' => $canReport['can_report'],
                'reason' => $canReport['reason']
            ]);

        } catch (Exception $e) {
            Log::error('Error checking can report: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التحقق من إمكانية الإبلاغ'
            ], 500);
        }
    }

    /**
     * API: التحقق من إغلاق المحادثة
     */
    public function checkConversationStatus($conversationId): JsonResponse
    {
        try {
            $isClosed = ClosedConversation::isConversationClosed($conversationId);
            $closureReason = null;
            
            if ($isClosed) {
                $closureReason = ClosedConversation::getClosureReason($conversationId);
            }
            
            return response()->json([
                'success' => true,
                'is_closed' => $isClosed,
                'closure_reason' => $closureReason,
                'message' => $isClosed ? 'تم إنهاء هذه المحادثة من قبل فريق دعم مرسوم' : null
            ]);

        } catch (Exception $e) {
            Log::error('Error checking conversation status: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التحقق من حالة المحادثة'
            ], 500);
        }
    }

    /**
     * API: الحصول على إحصائيات الإبلاغات
     */
    public function getStatistics(): JsonResponse
    {
        try {
            $stats = [
                'total_reports' => Report::count(),
                'pending_reports' => Report::byStatus(Report::STATUS_PENDING)->count(),
                'under_review_reports' => Report::byStatus(Report::STATUS_UNDER_REVIEW)->count(),
                'resolved_reports' => Report::byStatus(Report::STATUS_RESOLVED)->count(),
                'dismissed_reports' => Report::byStatus(Report::STATUS_DISMISSED)->count(),
                'closed_conversations' => ClosedConversation::count(),
                'recent_reports' => Report::recent(7)->count(),
            ];
            
            return response()->json([
                'success' => true,
                'data' => $stats
            ]);

        } catch (Exception $e) {
            Log::error('Error getting report statistics: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء جلب الإحصائيات'
            ], 500);
        }
    }

    /**
     * API: الحصول على إبلاغات المستخدم
     */
    public function getUserReports(): JsonResponse
    {
        try {
            $userId = Auth::id();
            $reports = Report::where('reporter_id', $userId)
                ->with(['reviewer'])
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($report) {
                    return [
                        'id' => $report->id,
                        'conversation_id' => $report->conversation_id,
                        'reason' => $report->reason,
                        'status' => $report->status,
                        'status_name' => $report->status_name,
                        'admin_notes' => $report->admin_notes,
                        'conversation_closed' => $report->conversation_closed,
                        'created_at' => $report->created_at->format('Y-m-d H:i:s'),
                        'reviewed_at' => $report->reviewed_at ? $report->reviewed_at->format('Y-m-d H:i:s') : null,
                        'reviewer_name' => $report->reviewer ? $report->reviewer->name : null,
                    ];
                });
            
            return response()->json([
                'success' => true,
                'data' => $reports
            ]);

        } catch (Exception $e) {
            Log::error('Error getting user reports: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء جلب إبلاغاتك'
            ], 500);
        }
    }
}



