<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Ad;
use App\Models\Category;
use App\Models\User;

class OrderController extends Controller
{
    /**
     * جلب جميع الصفقات مع تفاصيل الإعلان والمستخدم صاحب الإعلان.
     */
  public function index(Request $request)
{
    // جلب الفئات من قاعدة البيانات (أو يمكن استخدام طريقة أخرى لجلب الفئات)
    $categories = Order::distinct()->pluck('category');

    // جلب الإعلانات بناءً على الفئة المختارة أو جميع الفئات
    $category = $request->get('category', 'all'); // قيمة 'all' تعني جميع الفئات

    if ($category == 'all') {
        $orders = Order::with(['ad', 'ad.owner'])
            ->orderByDesc('updated_at')
            ->get();
    } else {
        $orders = Order::with(['ad', 'ad.owner'])
            ->where('category', $category)
            ->orderByDesc('updated_at')
            ->get();
    }

    if ($request->wantsJson()) {
        return response()->json($orders);
    }

    return view('instructor.deals.index', compact('orders', 'categories'));
}

public function create()
{
    return view('instructor.deals.create'); // أو اسم صفحة الـ Blade اللي فيها النموذج
}


public function myDeals()
{
    $orders = Order::where('user_id', auth()->id())->get(); 
    $categories = Category::all();
    return view('instructor.deals.my_deals', compact('orders', 'categories'));
}

    /**
     * إنشاء صفقة جديدة مع إنشاء إعلان تلقائيًا.
     */
public function store(Request $request)
{
    // ✅ التحقق من صحة البيانات المدخلة
    $request->validate([
        'user_id' => 'required|integer|exists:users,id',
        'title' => 'required|string|max:255',
        'description' => 'required|string',
        'category' => 'required|string|max:100',
        'price' => 'required|numeric',
    ]);

    $status = $request->status ?? 'قيد المراجعة';

    // ✅ إنشاء الإعلان المرتبط بالصفقة
    $ad = Ad::create([
        'user_id' => $request->user_id,
        'title' => $request->title,
        'description' => $request->description,
    ]);

    // ✅ إنشاء الصفقة الجديدة باستخدام الإعلان الذي تم إنشاؤه
    $order = Order::create([
        'user_id' => $request->user_id,
        'ad_id' => $ad->id,
        'title' => $request->title,
        'description' => $request->description,
        'category' => $request->category,
        'price' => $request->price,
        'status' => $status,
    ]);

return response()->json([
        'success' => true,
        'message' => 'تمت إضافة الصفقة بنجاح'
    ]);
}

    /**
     * عرض تفاصيل صفقة محددة.
     */
    public function show(Request $request, $id)
{
    $order = Order::with(['ad', 'ad.owner'])->find($id);

    if (!$order) {
        if ($request->wantsJson()) {
            return response()->json(['message' => 'الصفقة غير موجودة'], 404);
        } else {
            return abort(404, 'الصفقة غير موجودة');
        }
    }

    // لو المستخدم طلب JSON (من التطبيق مثلاً)
    if ($request->wantsJson()) {
        return response()->json($order);
    }

    // لو المستخدم يطلب HTML (من الموقع)
    return view('instructor.deals.show', compact('order'));
}


    /**
     * تحديث بيانات صفقة معينة.
     */
    public function update(Request $request, $id)
{
    $order = Order::find($id);

    if (!$order) {
        return redirect()->route('admin.orders.index')->with('status', [
            'type' => 'danger',
            'message' => 'الصفقة غير موجودة'
        ]);
    }

    // ✅ التحقق من صحة البيانات المدخلة
    $request->validate([
        'title' => 'sometimes|string|max:255',
        'description' => 'sometimes|string',
        'category' => 'sometimes|string|max:100',
        'price' => 'sometimes|numeric',
        'status' => 'sometimes|string',
    ]);

    // ✅ تحديث البيانات
    $order->update($request->all());

    return redirect()->route('admin.orders.index')->with('status', [
        'type' => 'success',
        'message' => 'تم تحديث الصفقة بنجاح'
    ]);
}

public function edit($id)
{
    // جلب الصفقة بناءً على الـ ID
    $order = Order::find($id);
    $categories = Category::all();
   
    // إذا لم توجد الصفقة
    if (!$order) {
        return redirect()->route('instructor.deals.index')->with('status', [
            'type' => 'danger',
            'message' => 'الصفقة غير موجودة'
        ]);
    }

    // عرض صفحة التعديل وتمرير البيانات
    return response()->json([
        'success' => true,
        'message' => 'تمت تعديل  الصفقة بنجاح'
    ]);
}


public function uppdate(Request $request, $id)
{
    // جلب الصفقة بناءً على الـ ID
    $order = Order::find($id);

    if (!$order) {
        return redirect()->route('instructor.deals.index')->with('status', [
            'type' => 'danger',
            'message' => 'الصفقة غير موجودة'
        ]);
    }

    // التحقق من صحة البيانات المدخلة (دون الحاجة إلى التحقق من الحالة لأننا سنجعلها "قيد المراجعة" دائمًا)
    $request->validate([
        'title' => 'required|string|max:255',
        'description' => 'required|string',
        'category' => 'required|string|max:100',
        'price' => 'required|numeric',
    ]);

    // تحديث البيانات مع تعيين الحالة تلقائيًا إلى "قيد المراجعة"
    $order->update([
        'title' => $request->title,
        'description' => $request->description,
        'category' => $request->category,
        'price' => $request->price,
        'status' => 'قيد المراجعة', // تعيين الحالة بشكل ثابت
    ]);

    // إعادة توجيه المستخدم مع رسالة نجاح
    return redirect()->route('instructor.deals.index')->with('status', [
        'type' => 'success',
        'message' => 'تم تحديث الصفقة بنجاح'
    ]);
}

    /**
     * حذف صفقة معينة مع حذف الإعلان المرتبط بها.
     */
 public function destroy($id)
{
    $order = Order::where('id', $id)
                  ->where('user_id', auth()->id())
                  ->first();

    if (!$order) {
        return response()->json([
            'success' => false,
            'message' => 'الصفقة غير موجودة أو ليس لديك صلاحية لحذفها'
        ], 404);
    }

    $order->ad()->delete();
    $order->delete();

    return response()->json([
        'success' => true,
        'message' => 'تم حذف الصفقة بنجاح'
    ]);
}

}