<?php

namespace App\Http\Controllers;

use App\Models\Conversation;
use App\Models\ChatMessage; // Changed from Message to ChatMessage
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ConversationController extends Controller
{
    /**
     * Display a listing of the user\'s conversations.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();

        // Get all conversations involving the current user
        $conversations = Conversation::where("user_one_id", $user->id)
            ->orWhere("user_two_id", $user->id)
            ->with(["userOne", "userTwo", "lastMessage.sender"])
            ->orderByDesc(
                ChatMessage::select("created_at") // Changed from Message to ChatMessage
                    ->whereColumn("conversation_id", "conversations.id")
                    ->latest()
                    ->take(1)
            )
            ->get();

        return view("conversations.index", compact("conversations"));
    }

    /**
     * Display the specified conversation.
     *
     * @param  \App\Models\Conversation  $conversation
     * @return \Illuminate\View\View
     */
    public function show(Conversation $conversation)
    {
        $user = Auth::user();

        // Ensure the current user is part of this conversation
        if ($conversation->user_one_id !== $user->id && $conversation->user_two_id !== $user->id) {
            abort(403, "Unauthorized action.");
        }

        // Mark all unread messages in this conversation as read for the current user
        $conversation->chatMessages() // Changed from messages() to chatMessages()
            ->where("read", false)
            ->where("sender_id", "!=", $user->id)
            ->update(["read" => true]);

        // Load messages for the conversation
        $messages = $conversation->chatMessages()->with("sender")->orderBy("created_at", "asc")->get(); // Changed from messages() to chatMessages()

        return view("conversations.show", compact("conversation", "messages"));
    }

    /**
     * Store a newly created message in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Conversation  $conversation
     * @return \Illuminate\Http\RedirectResponse
     */
    public function sendMessage(Request $request, Conversation $conversation)
    {
        $user = Auth::user();

        // Ensure the current user is part of this conversation
        if ($conversation->user_one_id !== $user->id && $conversation->user_two_id !== $user->id) {
            return response()->json(["message" => "Unauthorized"], 403);
        }

        $request->validate([
            "body" => "required|string|max:1000",
        ]);

        $message = $conversation->chatMessages()->create([ // Changed from messages() to chatMessages()
            "sender_id" => $user->id,
            "body" => $request->body,
            "read" => false, // New messages are unread by default
        ]);

        // Update last_message_id in conversation
        $conversation->update(["last_message_id" => $message->id]);

        return response()->json(["message" => "Message sent successfully", "data" => $message->load("sender")], 201);
    }

    /**
     * Start a new conversation with a specific user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function startConversation(Request $request)
    {
        $request->validate([
            "receiver_id" => "required|exists:users,id|not_in:" . Auth::id(),
        ]);

        $receiver = User::findOrFail($request->receiver_id);
        $user = Auth::user();

        $conversation = $user->getOrCreateConversationWith($receiver);

        return redirect()->route("conversations.show", $conversation->id);
    }

    /**
     * Get messages for a specific conversation (API).
     *
     * @param  \App\Models\Conversation  $conversation
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMessages(Conversation $conversation)
    {
        $user = Auth::user();

        // Ensure the current user is part of this conversation
        if ($conversation->user_one_id !== $user->id && $conversation->user_two_id !== $user->id) {
            return response()->json(["message" => "Unauthorized"], 403);
        }

        // Mark all unread messages in this conversation as read for the current user
        $conversation->chatMessages() // Changed from messages() to chatMessages()
            ->where("read", false)
            ->where("sender_id", "!=", $user->id)
            ->update(["read" => true]);

        $messages = $conversation->chatMessages()->with("sender")->orderBy("created_at", "asc")->get(); // Changed from messages() to chatMessages()

        return response()->json(["messages" => $messages]);
    }

    /**
     * Get unread message count for the current user (API).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUnreadCount()
    {
        $user = Auth::user();

        $unreadCount = ChatMessage::whereHas("conversation", function ($query) use ($user) {
            $query->where("user_one_id", $user->id)->orWhere("user_two_id", $user->id);
        })
        ->where("read", false)
        ->where("sender_id", "!=", $user->id)
        ->count();

        return response()->json(["unread_count" => $unreadCount]);
    }
}

