<?php 

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Job;
use App\Models\JobApplication;
use App\Models\User;
use Carbon\Carbon;

class JobController extends Controller
{
    public function list()
    {
        return Job::latest()->get();
    } 

public function checkSubscription($user_id)
{
    $user = User::find($user_id);
    if (!$user) return response()->json(['subscribed' => false]);

    $expiresAt = $user->job_subscription_expires_at 
        ? Carbon::parse($user->job_subscription_expires_at)
        : null;

    return response()->json([
        'subscribed' => $expiresAt && $expiresAt->isFuture()
    ]);
}

public function hasApplied(Request $request, $jobId)
{
    $user = auth()->user();

    if (!$user) {
        return response()->json(['message' => 'Unauthorized'], 401);
    }

    $hasApplied = Application::where('user_id', $user->id)
                             ->where('job_id', $jobId)
                             ->exists();

    return response()->json(['applied' => $hasApplied]);
}

    public function apply(Request $request)
{
    $request->validate([
        'job_id'       => 'required|exists:jobs,id',
        'user_id'      => 'required|exists:users,id',
        'cover_letter' => 'nullable|string',
        'cv_file'      => 'required|file|mimes:pdf,doc,docx|max:2048',
    ]);

    $path = $request->file('cv_file')->store('cvs', 'public');

    JobApplication::create([
        'job_id'       => $request->job_id,
        'user_id'      => $request->user_id,
        'cover_letter' => $request->cover_letter,
        'cv_file'      => $path, // ✅ يتوافق مع اسم العمود
    ]);

    return response()->json(['message' => 'Application sent successfully'], 201);
}
    
    public function index()
    {
        $jobs = Job::latest()->paginate(10);
        return view('admin.jobs.index', compact('jobs'));
    }

    public function create()
    {
        return view('admin.jobs.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'company' => 'required|string',
            'location' => 'required|string',
            'description' => 'required|string',
            'salary' => 'nullable|numeric', // ✅ حل المشكلة
            'job_type' => 'required|string', 
        ]);

        Job::create($request->only(['title', 'company', 'location', 'description', 'salary', 'job_type']));

        return redirect()->route('admin.jobs.index')->with('success', 'Job created successfully.');
    }

    public function edit($id)
    {
        $job = Job::findOrFail($id);
        return view('admin.jobs.edit', compact('job'));
    }

    public function update(Request $request, $id)
    {
        $job = Job::findOrFail($id);
        $job->update($request->only(['title', 'company', 'location', 'description', 'salary', 'job_type']));

        return redirect()->route('admin.jobs.index')->with('success', 'Job updated successfully.');
    }

    public function destroy($id)
    {
        Job::destroy($id);
        return redirect()->route('admin.jobs.index')->with('success', 'Job deleted successfully.');
    }
    
    public function showApplications($id)
{
    $job = Job::findOrFail($id);
    $applications = $job->applications()->with('user')->get();

    return view('admin.jobs.application', compact('job', 'applications'));
}

}

