<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DealsMessages;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class DealsMessagesController extends Controller
{
    /**
     * عرض قائمة المحادثات للمستخدم الحالي
     */
    public function index()
    {
        $userId = Auth::id();

        // جلب المحادثات الفريدة بناءً على thread_id
        $conversations = DealsMessages::where(function ($query) use ($userId) {
                $query->where('sender_id', $userId)
                      ->orWhere('receiver_id', $userId);
            })
            ->select(
                'thread_id',
                DB::raw('ANY_VALUE(id) as sample_id'),
                DB::raw('ANY_VALUE(sender_id) as sender_id'),
                DB::raw('ANY_VALUE(receiver_id) as receiver_id'),
                DB::raw('MAX(created_at) as last_activity')
            )
            ->groupBy('thread_id')
            ->orderBy('last_activity', 'desc')
            ->get();

        // إضافة معلومات إضافية لكل محادثة
        $conversations->each(function ($conversation) use ($userId) {
            // تحديد المستخدم الآخر في المحادثة
            $otherUserId = $conversation->sender_id == $userId ? $conversation->receiver_id : $conversation->sender_id;
            $conversation->other_user = User::find($otherUserId);
            
            // جلب آخر رسالة في المحادثة
            $conversation->last_message = DealsMessages::where('thread_id', $conversation->thread_id)
                ->orderBy('created_at', 'desc')
                ->first();
            
            // التحقق من وجود رسائل غير مقروءة
            $conversation->has_unread = DealsMessages::where('thread_id', $conversation->thread_id)
                ->where('receiver_id', $userId)
                ->where('read', false)
                ->exists();
        });

        return view('instructor.conversations.index', compact('conversations'));
    }

    /**
     * عرض محادثة محددة
     */
    public function show($thread_id)
    {
        $userId = Auth::id();

        // التحقق من أن المستخدم جزء من هذه المحادثة
        $userInConversation = DealsMessages::where('thread_id', $thread_id)
            ->where(function ($query) use ($userId) {
                $query->where('sender_id', $userId)
                      ->orWhere('receiver_id', $userId);
            })
            ->exists();

        if (!$userInConversation) {
            abort(403, 'غير مصرح لك بالوصول إلى هذه المحادثة');
        }

        // جلب جميع الرسائل في المحادثة
        $messages = DealsMessages::where('thread_id', $thread_id)
            ->with(['sender', 'receiver'])
            ->orderBy('created_at', 'asc')
            ->get();

        if ($messages->isEmpty()) {
            abort(404, 'المحادثة غير موجودة');
        }

        // تحديد المستخدم الآخر في المحادثة
        $firstMessage = $messages->first();
        $otherUserId = $firstMessage->sender_id == $userId ? $firstMessage->receiver_id : $firstMessage->sender_id;
        $otherUser = User::find($otherUserId);

        // تحديد الرسائل كمقروءة للمستخدم الحالي
        DealsMessages::where('thread_id', $thread_id)
            ->where('receiver_id', $userId)
            ->where('read', false)
            ->update(['read' => true]);

        return view('instructor.conversations.show', compact('messages', 'thread_id', 'otherUser'));
    }

    /**
     * إرسال رسالة جديدة
     */
    public function sendMessage(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'thread_id' => 'required|integer',
            'message' => 'required|string|max:1000'
        ]);

        if ($validator->fails()) {
            if ($request->wantsJson()) {
                return response()->json(['error' => $validator->errors()], 400);
            }
            return back()->withErrors($validator)->withInput();
        }

        $userId = Auth::id();
        $threadId = $request->thread_id;

        // التحقق من أن المستخدم جزء من هذه المحادثة
        $existingMessage = DealsMessages::where('thread_id', $threadId)
            ->where(function ($query) use ($userId) {
                $query->where('sender_id', $userId)
                      ->orWhere('receiver_id', $userId);
            })
            ->first();

        if (!$existingMessage) {
            if ($request->wantsJson()) {
                return response()->json(['error' => 'غير مصرح لك بإرسال رسائل في هذه المحادثة'], 403);
            }
            return back()->withErrors(['error' => 'غير مصرح لك بإرسال رسائل في هذه المحادثة']);
        }

        // تحديد المستقبل (المستخدم الآخر في المحادثة)
        $receiverId = $existingMessage->sender_id == $userId ? $existingMessage->receiver_id : $existingMessage->sender_id;

        // إنشاء الرسالة الجديدة
        $message = DealsMessages::create([
            'thread_id' => $threadId,
            'sender_id' => $userId,
            'receiver_id' => $receiverId,
            'message' => $request->message,
            'read' => false
        ]);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'تم إرسال الرسالة بنجاح',
                'data' => [
                    'id' => $message->id,
                    'thread_id' => $message->thread_id,
                    'sender_id' => $message->sender_id,
                    'receiver_id' => $message->receiver_id,
                    'message' => $message->message,
                    'read' => $message->read,
                    'created_at' => $message->created_at->toDateTimeString(),
                    'sender' => $message->sender
                ]
            ], 201);
        }

        return back()->with('success', 'تم إرسال الرسالة بنجاح');
    }

    /**
     * بدء محادثة جديدة مع مستخدم
     */
    public function startConversation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'receiver_id' => 'required|integer|exists:users,id|not_in:' . Auth::id()
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $senderId = Auth::id();
        $receiverId = $request->receiver_id;

        // البحث عن محادثة موجودة بين المستخدمين
        $existingThread = DealsMessages::where(function ($query) use ($senderId, $receiverId) {
                $query->where('sender_id', $senderId)->where('receiver_id', $receiverId);
            })
            ->orWhere(function ($query) use ($senderId, $receiverId) {
                $query->where('sender_id', $receiverId)->where('receiver_id', $senderId);
            })
            ->first();

        if ($existingThread) {
            // إعادة توجيه إلى المحادثة الموجودة
            return redirect()->route('conversations.show', ['thread_id' => $existingThread->thread_id]);
        }

        // إنشاء محادثة جديدة
        $threadId = time() . rand(1000, 9999); // استخدام timestamp + رقم عشوائي لتجنب التضارب

        $message = DealsMessages::create([
            'thread_id' => $threadId,
            'sender_id' => $senderId,
            'receiver_id' => $receiverId,
            'message' => 'مرحبا، كيف يمكنني مساعدتك؟',
            'read' => false
        ]);

        return redirect()->route('conversations.show', ['thread_id' => $threadId]);
    }

    /**
     * جلب الرسائل في محادثة (API)
     */
    public function getMessages($thread_id)
    {
        $userId = Auth::id();

        // التحقق من أن المستخدم جزء من هذه المحادثة
        $userInConversation = DealsMessages::where('thread_id', $thread_id)
            ->where(function ($query) use ($userId) {
                $query->where('sender_id', $userId)
                      ->orWhere('receiver_id', $userId);
            })
            ->exists();

        if (!$userInConversation) {
            return response()->json(['error' => 'غير مصرح لك بالوصول إلى هذه المحادثة'], 403);
        }

        // تحديد الرسائل كمقروءة
        DealsMessages::where('thread_id', $thread_id)
            ->where('receiver_id', $userId)
            ->where('read', false)
            ->update(['read' => true]);

        $messages = DealsMessages::where('thread_id', $thread_id)
            ->with(['sender'])
            ->orderBy('created_at', 'asc')
            ->get();

        return response()->json(['messages' => $messages]);
    }

    /**
     * جلب عدد الرسائل غير المقروءة للمستخدم الحالي
     */
    public function getUnreadCount()
    {
        $userId = Auth::id();

        $unreadCount = DealsMessages::where('receiver_id', $userId)
            ->where('read', false)
            ->count();

        return response()->json(['unread_count' => $unreadCount]);
    }

    /**
     * تحديد رسائل محادثة كمقروءة
     */
    public function markAsRead($thread_id)
    {
        $userId = Auth::id();

        DealsMessages::where('thread_id', $thread_id)
            ->where('receiver_id', $userId)
            ->where('read', false)
            ->update(['read' => true]);

        return response()->json(['message' => 'تم تحديد الرسائل كمقروءة']);
    }

    // الوظائف القديمة للتوافق مع النظام الحالي
    public function getUserThreads()
    {
        $userId = Auth::id();

        $threads = DealsMessages::where(function ($query) use ($userId) {
                $query->where('sender_id', $userId)
                      ->orWhere('receiver_id', $userId);
            })
            ->groupBy('thread_id')
            ->select(
                'thread_id',
                DB::raw('ANY_VALUE(sender_id) as sender_id'),
                DB::raw('ANY_VALUE(receiver_id) as receiver_id'),
                DB::raw('ANY_VALUE(message) as message'),
                DB::raw('MAX(created_at) as created_at')
            )
            ->orderBy('created_at', 'desc')
            ->get();

        $threads->each(function ($thread) use ($userId) {
            $otherUserId = $thread->sender_id == $userId ? $thread->receiver_id : $thread->sender_id;
            $otherUser = User::find($otherUserId);
            $thread->receiver_name = $otherUser ? $otherUser->name : null;
            $thread->receiver_photo = $otherUser ? $otherUser->photo : null;
        });
        
        return response()->json($threads);
    }

    public function openOrCreateConversation($receiver_id)
    {
        $sender_id = auth()->id();

        $existing = DealsMessages::where(function ($query) use ($sender_id, $receiver_id) {
            $query->where('sender_id', $sender_id)
                  ->where('receiver_id', $receiver_id);
        })->orWhere(function ($query) use ($sender_id, $receiver_id) {
            $query->where('sender_id', $receiver_id)
                  ->where('receiver_id', $sender_id);
        })->first();

        if ($existing) {
            $thread_id = $existing->thread_id;
        } else {
            $thread_id = time() . rand(1000, 9999);

            DealsMessages::create([
                'thread_id' => $thread_id,
                'sender_id' => $sender_id,
                'receiver_id' => $receiver_id,
                'message' => 'مرحبا، كيف يمكنني مساعدتك؟',
                'read' => false,
            ]);
        }

        return redirect()->route('instructor.conversations.show', ['thread_id' => $thread_id]);
    }

public function openapp($receiver_id)
{
    $sender_id = auth()->id();

    if ($receiver_id == $sender_id) {
        return response()->json([
            'error' => 'لا يمكنك بدء محادثة مع نفسك'
        ], 400);
    }

    $exists = User::where('id', $receiver_id)->exists();
    if (!$exists) {
        return response()->json([
            'error' => 'المستخدم غير موجود'
        ], 404);
    }

    $existing = DealsMessages::where(function ($query) use ($sender_id, $receiver_id) {
        $query->where('sender_id', $sender_id)
              ->where('receiver_id', $receiver_id);
    })->orWhere(function ($query) use ($sender_id, $receiver_id) {
        $query->where('sender_id', $receiver_id)
              ->where('receiver_id', $sender_id);
    })->first();

    if ($existing) {
        return response()->json([
            'thread_id' => $existing->thread_id,
            'status' => 'exists'
        ]);
    }

    $thread_id = time() . rand(1000, 9999);

    DealsMessages::create([
        'thread_id' => $thread_id,
        'sender_id' => $sender_id,
        'receiver_id' => $receiver_id,
        'message' => 'مرحبا، كيف يمكنني مساعدتك؟',
        'read' => false,
    ]);

    return response()->json([
        'thread_id' => $thread_id,
        'status' => 'created'
    ]);
}
}


