<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Certificate;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\LessonProgress;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;

class CertificateController extends Controller
{
    // ✅ حالة الشهادة
    public function getCertificateStatus(Request $request)
    {
        $request->validate([
            'course_id' => 'required|exists:courses,id',
        ]);

        $user = $request->user();
        $courseId = $request->course_id;

        $courseProgress = $this->getCourseProgress($user->id, $courseId);

        $certificate = Certificate::where('course_id', $courseId)
            ->where('user_id', $user->id)
            ->first();

        if ($certificate && $courseProgress >= 100) {
            return response()->json([
                'status' => 'completed',
                'progress' => round($courseProgress),
                'message' => 'تهانينا! يمكنك تحميل شهادتك.',
                'certificate_link' => route('certificate', ['identifier' => $certificate->identifier]),
            ]);
        } else {
            return response()->json([
                'status' => 'in_progress',
                'progress' => round($courseProgress),
                'message' => 'تابع التقدم! أكمل الدورة لتحصل على شهادتك.',
            ]);
        }
    }

    // ✅ إنشاء شهادة
    public function createCertificate(Request $request)
    {
        $request->validate([
            'course_id' => 'required|exists:courses,id',
        ]);

        $user = $request->user();
        $courseId = $request->course_id;

        $existingCertificate = Certificate::where('user_id', $user->id)
            ->where('course_id', $courseId)
            ->first();

        if ($existingCertificate) {
            return response()->json([
                'status' => 'exists',
                'message' => 'الشهادة موجودة بالفعل.',
                'certificate' => $existingCertificate,
            ], 200);
        }

        $identifier = strtoupper(uniqid('CERT-'));

        $certificate = Certificate::create([
            'user_id' => $user->id,
            'course_id' => $courseId,
            'identifier' => $identifier,
        ]);

        return response()->json([
            'status' => 'created',
            'message' => 'تم إنشاء الشهادة بنجاح.',
            'certificate' => $certificate,
        ], 201);
    }

    // ✅ جلب شهادات المستخدم
    public function getUserCertificates(Request $request)
    {
        $user = $request->user();

        $certificates = Certificate::with('course')
            ->where('user_id', $user->id)
            ->get();

        return response()->json([
            'status' => 'success',
            'certificates' => $certificates,
        ], 200);
    }

    // ✅ تحميل شهادة PDF
    public function downloadCertificate($identifier)
    {
        $certificate = Certificate::with(['course', 'user'])
            ->where('identifier', $identifier)
            ->first();

        if (!$certificate) {
            return response()->json([
                'status' => 'error',
                'message' => 'الشهادة غير موجودة.',
            ], 404);
        }

        $pdf = Pdf::loadView('pdf.certificate', [
            'certificate' => $certificate
        ]);

        return $pdf->download('certificate-' . $identifier . '.pdf');
    }

    // ✅ إنشاء PDF للشهادة وحفظها في التخزين
    public function generateCertificate(Request $request)
    {
        $request->validate([
            'course_id' => 'required|exists:courses,id',
        ]);

        $user = $request->user();

        $certificate = Certificate::firstOrCreate(
            [
                'user_id' => $user->id,
                'course_id' => $request->course_id
            ],
            [
                'identifier' => strtoupper(uniqid('CERT-'))
            ]
        );

        $pdf = Pdf::loadView('pdf.certificate', [
            'certificate' => $certificate
        ]);

        $fileName = 'certificate-' . $certificate->identifier . '.pdf';
        $filePath = 'certificates/' . $fileName;

        Storage::disk('public')->put($filePath, $pdf->output());

        $pdfUrl = asset('storage/' . $filePath);

        return response()->json([
            'status' => 'success',
            'message' => 'تم إنشاء شهادة PDF بنجاح.',
            'pdf_url' => $pdfUrl,
        ]);
    }

    // ✅ حساب التقدم
    private function getCourseProgress($userId, $courseId)
    {
        $totalLessons = Lesson::where('course_id', $courseId)->count();
        $completedLessons = LessonProgress::where('course_id', $courseId)
            ->where('user_id', $userId)
            ->where('is_completed', true)
            ->count();

        if ($totalLessons == 0) return 0;

        return ($completedLessons / $totalLessons) * 100;
    }
}
