<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\WalletTransaction;
use App\Models\WithdrawalRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class WalletAdminController extends Controller
{
    // عرض لوحة تحكم المحفظة
    public function dashboard()
    {
        $stats = [
            'total_balance' => User::sum('wallet_balance') + User::sum('frozen_balance'),
            'available_balance' => User::sum('wallet_balance'),
            'frozen_balance' => User::sum('frozen_balance'),
            'total_transactions' => WalletTransaction::count(),
            'pending_withdrawals' => WithdrawalRequest::where('status', 'pending')->count(),
        ];

        return view('admin.wallet.dashboard', compact('stats'));
    }

    // عرض جميع المعاملات
    public function transactions(Request $request)
    {
        $search = $request->input('search');
        
        $transactions = WalletTransaction::with(['sender', 'receiver'])
            ->when($search, function($query) use ($search) {
                return $query->where('transaction_id', 'like', "%$search%")
                    ->orWhere('amount', $search)
                    ->orWhereHas('sender', function($q) use ($search) {
                        $q->where('name', 'like', "%$search%")
                          ->orWhere('wallet_id', 'like', "%$search%");
                    })
                    ->orWhereHas('receiver', function($q) use ($search) {
                        $q->where('name', 'like', "%$search%")
                          ->orWhere('wallet_id', 'like', "%$search%");
                    });
            })
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.wallet.transactions', compact('transactions', 'search'));
    }

    // عرض تفاصيل معاملة
    public function showTransaction($id)
    {
        $transaction = WalletTransaction::with(['sender', 'receiver'])->findOrFail($id);
        return view('admin.wallet.transaction-details', compact('transaction'));
    }

    // عرض نموذج إضافة رصيد
    public function showAddBalanceForm()
    {
        return view('admin.wallet.add-balance');
    }

    // معالجة إضافة رصيد
    public function addBalance(Request $request)
{
    $request->validate([
        'user_id' => 'required|exists:users,id',
        'amount' => 'required|numeric|min:0.01',
        'transaction_type' => 'required|in:deposit,withdraw',
        'notes' => 'nullable|string',
    ]);

    $user = User::findOrFail($request->user_id);
    $admin = auth()->user();
    $amount = $request->amount;
    $transactionType = $request->transaction_type;

    DB::transaction(function () use ($user, $amount, $transactionType, $admin, $request) {
        // معالجة العملية حسب النوع
        if ($transactionType == 'withdraw') {
            if ($user->wallet_balance < $amount) {
                throw new \Exception('رصيد المستخدم لا يكفي لعملية السحب');
            }
            $user->wallet_balance -= $amount;
            $notes = $request->notes ?? 'سحب رصيد من قبل المشرف';
        } else {
            $user->wallet_balance += $amount;
            $notes = $request->notes ?? 'إضافة رصيد من قبل المشرف';
        }

        $user->save();

        // تسجيل المعاملة
        WalletTransaction::create([
            'transaction_id' => 'ADM' . Str::upper(Str::random(10)),
            'sender_id' => $transactionType == 'withdraw' ? $user->id : $admin->id,
            'receiver_id' => $transactionType == 'withdraw' ? $admin->id : $user->id,
            'amount' => $amount,
            'type' => $transactionType,
            'status' => 'completed',
            'notes' => $notes,
        ]);
    });

    return redirect()->route('admin.wallet.transactions')
        ->with('success', 'تم تنفيذ العملية بنجاح');
}

    // عرض طلبات السحب
    public function withdrawalRequests(Request $request)
    {
        $status = $request->input('status', 'pending');
        $search = $request->input('search');

        $requests = WithdrawalRequest::with(['user', 'admin'])
            ->when($status != 'all', function($query) use ($status) {
                return $query->where('status', $status);
            })
            ->when($search, function($query) use ($search) {
                return $query->where('request_id', 'like', "%$search%")
                    ->orWhere('amount', $search)
                    ->orWhereHas('user', function($q) use ($search) {
                        $q->where('name', 'like', "%$search%")
                          ->orWhere('wallet_id', 'like', "%$search%");
                    });
            })
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.wallet.withdrawal-requests', compact('requests', 'status', 'search'));
    }

    // عرض تفاصيل طلب السحب
    public function showWithdrawalRequest($id)
    {
        $request = WithdrawalRequest::with(['user', 'admin'])->findOrFail($id);
        return view('admin.wallet.withdrawal-details', compact('request'));
    }

    // الموافقة على طلب السحب
    public function approveWithdrawal(Request $request, $id)
    {
        $withdrawal = WithdrawalRequest::findOrFail($id);

        if ($withdrawal->status != 'pending') {
            return back()->with('error', 'لا يمكن الموافقة على طلب غير معلق');
        }

        $withdrawal->update([
            'status' => 'approved',
            'processed_by' => auth()->id(),
            'processed_at' => now(),
            'admin_notes' => $request->admin_notes,
        ]);

        return redirect()->route('admin.wallet.withdrawal-requests')
            ->with('success', 'تمت الموافقة على طلب السحب بنجاح');
    }

    // رفض طلب السحب
    public function rejectWithdrawal(Request $request, $id)
    {
        $withdrawal = WithdrawalRequest::findOrFail($id);
        $user = $withdrawal->user;

        if ($withdrawal->status != 'pending') {
            return back()->with('error', 'لا يمكن رفض طلب غير معلق');
        }

        DB::transaction(function () use ($withdrawal, $user, $request) {
            // إعادة المبلغ للمستخدم
            $user->wallet_balance += $withdrawal->amount;
            $user->save();

            // تحديث حالة الطلب
            $withdrawal->update([
                'status' => 'rejected',
                'processed_by' => auth()->id(),
                'processed_at' => now(),
                'admin_notes' => $request->admin_notes,
            ]);
        });

        return redirect()->route('admin.wallet.withdrawal-requests')
            ->with('success', 'تم رفض طلب السحب وإعادة المبلغ للمستخدم');
    }
}