<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class JobSubscriptionController extends Controller
{
    public function index()
    {
        $now = Carbon::now();

        $activeSubscriptions = User::whereNotNull('job_subscription_expires_at')
            ->where('job_subscription_expires_at', '>', $now)
            ->orderBy('job_subscription_expires_at', 'desc')
            ->paginate(10);

        $expiredSubscriptions = User::whereNotNull('job_subscription_expires_at')
            ->where('job_subscription_expires_at', '<=', $now)
            ->orderBy('job_subscription_expires_at', 'desc')
            ->paginate(10);

        $stats = [
            'total_active' => User::whereNotNull('job_subscription_expires_at')
                ->where('job_subscription_expires_at', '>', $now)
                ->count(),
            'total_expired' => User::whereNotNull('job_subscription_expires_at')
                ->where('job_subscription_expires_at', '<=', $now)
                ->count(),
            'expiring_soon' => User::whereNotNull('job_subscription_expires_at')
                ->where('job_subscription_expires_at', '>', $now)
                ->where('job_subscription_expires_at', '<=', $now->copy()->addDays(7))
                ->count(),
            'total_users' => User::count()
        ];

        return view('admin.jobs.subscriptions.index', compact('activeSubscriptions', 'expiredSubscriptions', 'stats'));
    }

    public function create()
    {
        return view('admin.jobs.subscriptions.create');
    }

    public function searchUsers(Request $request)
    {
        $query = $request->get('q');

        if (strlen($query) < 2) {
            return response()->json([]);
        }

        $users = User::where(function($q) use ($query) {
                $q->where('name', 'LIKE', "%{$query}%")
                  ->orWhere('email', 'LIKE', "%{$query}%");
            })
            ->select('id', 'name', 'email', 'job_subscription_expires_at')
            ->limit(10)
            ->get();

        return response()->json($users->map(function($user) {
            $expiresAt = $user->job_subscription_expires_at ? Carbon::parse($user->job_subscription_expires_at)->endOfDay() : null;
            return [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'current_expiry' => $expiresAt ? $expiresAt->format('Y-m-d') : null,
                'is_active' => $expiresAt && $expiresAt->isFuture(),
                'display_text' => $user->name . ' (' . $user->email . ')'
            ];
        }));
    }

    public function store(Request $request)
    {
        \Log::info('Subscription store request received', $request->all());

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'expires_at' => 'required|date|after:today',
            'subscription_type' => 'required|in:new,extend,renew'
        ]);

        try {
            DB::beginTransaction();

            $user = User::findOrFail($request->user_id);
            $expiresAt = Carbon::parse($request->expires_at)->endOfDay();

            \Log::info('Updating user subscription', [
                'user_id' => $user->id,
                'user_name' => $user->name,
                'expires_at' => $expiresAt->toDateTimeString()
            ]);

            $updated = $user->update([
                'job_subscription_expires_at' => $expiresAt
            ]);

            if (!$updated) {
                throw new \Exception('فشل في تحديث بيانات المستخدم');
            }

            \Log::info('User subscription updated successfully', [
                'user_id' => $user->id,
                'new_expiry' => $user->fresh()->job_subscription_expires_at
            ]);

            $this->logSubscriptionActivity($user, $request->subscription_type, $expiresAt);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث اشتراك المستخدم بنجاح',
                'user' => [
                    'name' => $user->name,
                    'email' => $user->email,
                    'expires_at' => $expiresAt->format('Y-m-d H:i:s'),
                    'expires_at_formatted' => $expiresAt->format('d/m/Y')
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            \Log::error('Validation error in subscription store', [
                'errors' => $e->errors(),
                'request' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطأ في البيانات المدخلة: ' . implode(', ', $e->validator->errors()->all())
            ], 422);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error in subscription store', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء تحديث الاشتراك: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($userId)
    {
        $user = User::findOrFail($userId);
        $expiresAt = $user->job_subscription_expires_at ? Carbon::parse($user->job_subscription_expires_at)->endOfDay() : null;

        return response()->json([
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'job_subscription_expires_at' => $expiresAt ? $expiresAt->toDateTimeString() : null,
            'is_active' => $expiresAt && $expiresAt->isFuture(),
            'days_remaining' => $expiresAt ? max(Carbon::now()->startOfDay()->diffInDays($expiresAt, false), 0) : null
        ]);
    }

    public function destroy($userId)
    {
        try {
            $user = User::findOrFail($userId);

            $user->update([
                'job_subscription_expires_at' => null
            ]);

            return response()->json([
                'success' => true,
                'message' => 'تم إلغاء اشتراك المستخدم بنجاح'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إلغاء الاشتراك: ' . $e->getMessage()
            ], 500);
        }
    }

    private function logSubscriptionActivity($user, $type, $expiresAt)
    {
        // اختياري لتسجيل السجلات
    }

    public function export(Request $request)
    {
        $type = $request->get('type', 'active');

        $query = User::whereNotNull('job_subscription_expires_at');

        if ($type === 'active') {
            $query->where('job_subscription_expires_at', '>', now());
        } elseif ($type === 'expired') {
            $query->where('job_subscription_expires_at', '<=', now());
        }

        $users = $query->orderBy('job_subscription_expires_at', 'desc')->get();

        return response()->json([
            'message' => 'ميزة التصدير ستكون متاحة قريباً',
            'count' => $users->count()
        ]);
    }
}
