<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Job;
use App\Models\JobApplication;
use Illuminate\Support\Facades\Auth;

class JobController extends Controller
{
    // الوظائف الموجودة مسبقاً...
    
    public function list()
    {
        $jobs = Job::latest()->paginate(10);
        return response()->json($jobs);
    }

    public function checkSubscription($user_id)
    {
        // منطق التحقق من الاشتراك
        // يجب تطبيق المنطق المناسب حسب نظامك
        return response()->json(['subscribed' => true]);
    }

    public function apply(Request $request)
    {
        $request->validate([
            'job_id' => 'required|exists:jobs,id',
            'cover_letter' => 'nullable|string',
            'cv_file' => 'required|file|mimes:pdf,doc,docx|max:5120', // 5MB max
        ]);

        $user = Auth::user();
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول للتقديم على الوظيفة'
            ], 401);
        }

        $jobId = $request->job_id;

        // التحقق من وجود تقديم مسبق
        $existingApplication = JobApplication::where('user_id', $user->id)
                                           ->where('job_id', $jobId)
                                           ->first();

        if ($existingApplication) {
            return response()->json([
                'success' => false,
                'message' => 'لقد قمت بالتقديم على هذه الوظيفة مسبقاً'
            ], 409); // Conflict
        }

        try {
            // رفع ملف السيرة الذاتية
            $cvFile = $request->file('cv_file');
            $cvFileName = time() . '_' . $user->id . '_' . $cvFile->getClientOriginalName();
            $cvPath = $cvFile->storeAs('cv_files', $cvFileName, 'public');

            // إنشاء التقديم
            JobApplication::create([
                'user_id' => $user->id,
                'job_id' => $jobId,
                'cover_letter' => $request->cover_letter,
                'cv_file_path' => $cvPath,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'تم إرسال طلب التقديم بنجاح'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إرسال طلب التقديم'
            ], 500);
        }
    }

    public function hasApplied($jobId)
    {
        $user = Auth::user();
        
        if (!$user) {
            return response()->json([
                'applied' => false,
                'message' => 'المستخدم غير مسجل الدخول'
            ], 200);
        }

        $hasApplied = JobApplication::where('user_id', $user->id)
                                  ->where('job_id', $jobId)
                                  ->exists();

        return response()->json([
            'applied' => $hasApplied,
            'message' => $hasApplied ? 'لقد قمت بالتقديم على هذه الوظيفة مسبقاً' : 'يمكنك التقديم على هذه الوظيفة'
        ]);
    }
}


