<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Dispute;
use App\Models\DealsMessages;
use App\Models\Message;
use Illuminate\Http\Request;

class DisputeController extends Controller
{
    public function index()
    {
        $disputes = Dispute::with(['dealMessage', 'reporter']) // العلاقة الصحيحة
            ->latest()
            ->paginate(10);

        return view('admin.disputes.index', [
            'disputes' => $disputes,
            'statuses' => Dispute::statuses()
        ]);
    }

    public function show(Dispute $dispute)
    {
        $messages = Message::where('deal_message_id', $dispute->deal_message_id)
            ->with('user')
            ->latest()
            ->get();

        return view('admin.disputes.show', [
            'dispute' => $dispute->load('reporter'),
            'messages' => $messages
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'deal_message_id' => 'required|exists:deals_messages,id',
            'reason' => 'nullable|string|max:500'
        ]);

        if (Dispute::where('deal_message_id', $validated['deal_message_id'])->active()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'يوجد إبلاغ نشط بالفعل لهذه المحادثة'
            ], 422);
        }

        $dispute = Dispute::create([
            'deal_message_id' => $validated['deal_message_id'],
            'reporter_id' => auth()->id(),
            'reason' => $validated['reason'] ?? null,
            'status' => 'pending'
        ]);

        DealsMessages::where('id', $validated['deal_message_id'])
            ->update(['has_dispute' => true]);

        return response()->json([
            'success' => true,
            'message' => 'تم الإبلاغ بنجاح',
            'dispute_id' => $dispute->id
        ]);
    }

    public function sendMessage(Request $request, Dispute $dispute)
    {
        if (auth()->user()->role !== 'admin') {
            return response()->json(['message' => 'غير مصرح'], 403);
        }

        $validated = $request->validate([
            'message' => 'required|string|max:1000'
        ]);

        $message = Message::create([
            'deal_message_id' => $dispute->deal_message_id,
            'user_id' => auth()->id(),
            'body' => $validated['message'],
            'is_admin' => true,
            'type' => 'dispute'
        ]);

        if ($dispute->status === 'pending') {
            $dispute->update([
                'status' => 'in_progress',
                'admin_id' => auth()->id()
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'تم إرسال الرسالة',
            'message_id' => $message->id
        ]);
    }

    public function resolve(Dispute $dispute)
    {
        if (auth()->user()->role !== 'admin') {
            return response()->json(['message' => 'غير مصرح'], 403);
        }

        $dispute->update([
            'status' => 'resolved',
            'resolved_at' => now()
        ]);

        DealsMessages::where('id', $dispute->deal_message_id)
            ->update(['has_dispute' => false]);

        return redirect()->route('admin.disputes.index')
            ->with('success', 'تم حل النزاع بنجاح');
    }
}