<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\UserBan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use App\Models\ScreenCaptureViolation;

class BanManagementController extends Controller
{
    public function index(Request $request)
    {
        $query = UserBan::with("user", "admin");

        if ($request->has("search") && $request->search != "") {
            $search = $request->search;
            $query->whereHas("user", function ($q) use ($search) {
                $q->where("name", "LIKE", "%" . $search . "%")
                  ->orWhere("email", "LIKE", "%" . $search . "%");
            });
        }

        $page_data["bans"] = $query->orderBy("created_at", "desc")->paginate(10);
        return view("admin.ban_management.index", $page_data);
    }

    public function create()
    {
        $page_data["users"] = User::where("role", "student")->get(); // Only ban students
        return view("admin.ban_management.create", $page_data);
    }

    public function store(Request $request)
    {
        $request->validate([
            "user_id" => "required|exists:users,id",
            "reason" => "required|string|max:255",
            "duration" => "nullable|integer|min:0", // Duration in minutes, 0 for permanent
            "ban_message" => "nullable|string",
            "ban_image_url" => "nullable|url",
        ]);

        $user = User::find($request->user_id);

        // Deactivate any existing active ban for this user
        UserBan::where("user_id", $user->id)->update(["is_active" => false]);

        $bannedUntil = null;
        if ($request->duration !== null && $request->duration > 0) {
            $bannedUntil = now()->addMinutes((int)$request->duration);
        }

        UserBan::create([
            "user_id" => $user->id,
            "banned_until" => $bannedUntil,
            "reason" => $request->reason,
            "admin_id" => Auth::id(),
            "is_active" => true,
            "ban_message" => $request->ban_message,
            "ban_image_url" => $request->ban_image_url,
        ]);

        Session::flash("success", "تم حظر المستخدم بنجاح.");
        return redirect()->route("admin.ban_management.index");
    }

    public function unban($id)
    {
        $ban = UserBan::findOrFail($id);
        $ban->update([
            "is_active" => false,
            "banned_until" => now(), // Set banned_until to now to mark as immediately unbanned
        ]);

        Session::flash("success", "تم إلغاء حظر المستخدم بنجاح.");
        return redirect()->route("admin.ban_management.index");
    }

    public function showViolations(Request $request)
    {
        $query = ScreenCaptureViolation::with("user");

        if ($request->has("search") && $request->search != "") {
            $search = $request->search;
            $query->whereHas("user", function ($q) use ($search) {
                $q->where("name", "LIKE", "%" . $search . "%")
                  ->orWhere("email", "LIKE", "%" . $search . "%");
            });
        }

        $page_data["violations"] = $query->orderBy("violation_timestamp", "desc")->paginate(10);
        return view("admin.ban_management.violations", $page_data);
    }

    /**
     * تتحقق مما إذا كان المستخدم محظورًا بشكل نشط.
     *
     * @param int $userId معرف المستخدم
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkBanStatus($userId)
    {
        // البحث عن حظر نشط للمستخدم
        $activeBan = UserBan::where('user_id', $userId)
                           ->where('is_active', true)
                           ->first(); // نستخدم first() هنا لأننا نحتاج للتحقق من banned_until

        $isBanned = false;
        if ($activeBan) {
            // إذا كان الحظر دائم (banned_until هو NULL) أو لم ينته بعد
            if ($activeBan->banned_until === null || $activeBan->banned_until->isFuture()) {
                $isBanned = true;
            }
        }

        return response()->json([
            'is_banned' => $isBanned
        ]);
    }

    public function reban(User $user)
    {
        // البحث عن آخر سجل حظر غير نشط للمستخدم
        $inactiveBan = UserBan::where('user_id', $user->id)
                            ->where('is_active', false)
                            ->latest()
                            ->first();

        if ($inactiveBan) {
            // تحديث السجل ليصبح نشط
            $inactiveBan->update([
                'is_active' => true,
                'banned_until' => NULL, // إذا كنت تريد أن يكون الحظر المعاد دائمًا
            ]);

            return back()->with('success', 'تم إعادة تفعيل الحظر للمستخدم.');
        } else {
            return back()->with('error', 'لا يوجد سجل حظر غير نشط لهذا المستخدم.');
        }
    }
}
