<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order; // تأكد من أن لديك موديل Order
use Illuminate\Support\Facades\Auth;

class AdminOrderController extends Controller
{
    /**
     * جلب جميع الطلبات (الصفقات) مع تفاصيل الإعلان والمستخدم
     */
    public function index()
    {
        $orders = Order::with(['ad', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(10); // تقسيم النتائج لـ 10 عناصر لكل صفحة

        return view('admin.orders.index', compact('orders'));
    }

    /**
     * إضافة طلب جديد (صفقة جديدة)
     */
    public function store(Request $request)
    {
        // التحقق من المصادقة
        if (!Auth::check()) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        // التحقق من صحة البيانات
        $validatedData = $request->validate([
            'ad_id' => 'required|integer|exists:ads,id',  // التأكد من أن الإعلان موجود
            'total_price' => 'required|numeric|min:1',
            'quantity' => 'required|integer|min:1',
            'notes' => 'nullable|string|max:500',
            'payment_method' => 'required|string|in:cash,credit_card,bank_transfer',
            'payment_status' => 'required|string|in:pending,completed,failed',
            'order_date' => 'required|date',
        ]);

        // إنشاء الطلب الجديد
        $order = Order::create([
            'user_id' => Auth::id(),
            'ad_id' => $validatedData['ad_id'],
            'status' => 'قيد المراجعة', // الحالة الافتراضية للطلب
            'total_price' => $validatedData['total_price'],
            'quantity' => $validatedData['quantity'],
            'notes' => $validatedData['notes'] ?? null,
            'payment_method' => $validatedData['payment_method'],
            'payment_status' => $validatedData['payment_status'],
            'order_date' => $validatedData['order_date'],
        ]);

        return response()->json([
            'message' => 'تمت إضافة الطلب بنجاح',
            'order' => $order
        ], 201);
    }

    /**
     * تحديث حالة الطلب
     */
    public function updateStatus(Request $request, $id)
{
    $request->validate([
        'status' => 'required|string|in:قيد المراجعة,موافق,مرفوض'
    ]);

    $order = Order::findOrFail($id);
    $order->status = $request->status;
    $order->save();

    return redirect()->back()->with('success', 'تم تحديث الحالة بنجاح!');

}

public function destroy($id)
{
    $order = Order::findOrFail($id);
    $order->delete();

    return redirect()->route('admin.orders.index')->with('success', 'تم حذف الصفقة بنجاح');
}

}
